package org.apache.zookeeper.server.persistence;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;

import com.google.common.cache.LoadingCache;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.server.DataTree;
import org.apache.lucene.analysis.KeywordAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.Fields;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriter.IndexReaderWarmer;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.SharedThreadPoolMergeScheduler;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.index.TieredMergePolicy;
import org.apache.lucene.store.BufferedIndexOutput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.Version;
import org.jctools.maps.NonBlockingHashMap;

import ru.yandex.concurrent.WeighableRunnable;
import ru.yandex.logger.PrefixedLogger;

public class LuceneQueue {
    private static final int HASH_MISS_CACHE_SIZE = 1000000;
//    private static final Logger LOG = LoggerFactory.getLogger(LuceneQueue.class);
    private static final String LARGEST_QUEUE_NODE =
        "queue99999999999999999999";
    private static final String LARGEST_NODE = "LARGEST_NODE:";
    private static final long NOT_INITIALIZED = Long.MIN_VALUE;
    private static final int DATA_OVERHEAD = 48;
    private static final int RAM_BUFFER_SIZE_MB = ramBufferSizeMB();

    private final PrefixedLogger logger;
//    private File storageDir;
    private File dataDir;
    private NIOFSDirectory dataDirectory;
    private IndexWriter indexWriter = null;
    private ReentrantReadWriteLock readLock = new ReentrantReadWriteLock();
    private ReentrantReadWriteLock indexLock = new ReentrantReadWriteLock();
    private volatile boolean initialized = false;
    private volatile boolean segmentsChanged = false;
    private final LoadingCache<LuceneStorage.CacheKey, byte[]> diskCache;
    private final long maxSegmentSize;
    private SharedThreadPoolMergeScheduler mergeScheduler;
    private TieredMergePolicy mergePolicy;
    private static final Set<String> storedFields = storedFieldsInit();
    private static final Set<String> indexedFields = indexedFieldsInit();
    private Map<String, String> commitData;
    private final NonBlockingHashMap<String, AtomicLong> largestDiskNodes =
    new NonBlockingHashMap<>();
    private final NonBlockingHashMap<String, Long> commitedLargestNodes =
    new NonBlockingHashMap<>();
    private final AtomicReference<MemoryPart> currentMemoryPart =
    new AtomicReference<>();
    private final AtomicReference<Thread> flushLock = new AtomicReference();
    private final ConcurrentHashMap<
        String,
        Comparator<AtomicReaderContext>> contextsComparators =
            new ConcurrentHashMap<>();
    private final Comparator<AtomicReaderContext> readerContextComparator =
        new ReaderContextComparator("hash");

    private static int ramBufferSizeMB() {
        String size = System.getProperty(
            "ru.yandex.lucene-queue.ram-buffer-size-mb",
            "600");
        return Integer.parseInt(size);
    }

    private static class ReaderFinished
        implements IndexReader.ReaderFinishedListener
    {
        private volatile boolean finished;
        private final IndexReader referenceReader;

        ReaderFinished(final IndexReader referenceReader) {
            this.referenceReader = referenceReader;
        }

        public boolean finished() {
            return finished;
        }

        @Override
        public void finished(final IndexReader reader) {
            if (reader != referenceReader) {
                try {
                    referenceReader.directory();
                } catch (Exception e) {
                    System.err.println("Reader: " + referenceReader
                        + " has been closed by another reader");
                    e.printStackTrace();
                    finished = true;
                }
                return;
            }
            reader.removeReaderFinishedListener(this);
            finished = true;
        }
    }

    private class MemoryPart {
        private final NonBlockingHashMap<String, Node> liveData =
            new NonBlockingHashMap<String, Node>();
        private final DoubleBarrelLRUCache<String, String> hashMiss =
            new DoubleBarrelLRUCache<>(HASH_MISS_CACHE_SIZE);
        private final IndexWriter indexWriter;
        private final AtomicReference<IndexReader> indexReader =
            new AtomicReference<>();
        private final ConcurrentLinkedQueue<Document> defferedData =
            new ConcurrentLinkedQueue<>();
        private final AtomicLong liveDataSize = new AtomicLong(0L);
        private final ConcurrentLinkedQueue<Runnable> onCommitCallbacks =
            new ConcurrentLinkedQueue<>();
        private final AtomicReference<ConcurrentSkipListSet<String>> sortedSet =
            new AtomicReference<>();
        private final AtomicBoolean sortedKeysInitialized = new AtomicBoolean(false);
        private final AtomicInteger requestCount = new AtomicInteger(0);
        private volatile AtomicReaderContext[] readers;
        private volatile boolean closed = false;
        private MemoryPart old = null;

        public MemoryPart(final IndexWriter indexWriter) {
            this.indexWriter = indexWriter;
        }

        public NonBlockingHashMap<String, Node> liveData() {
            return liveData;
        }

        public Map<String, Node> oldLiveData() {
            final MemoryPart old = this.old;
            if (old != null) {
                return old.liveData();
            }
            return Collections.emptyMap();
        }

        public AtomicReaderContext[] readers() {
            boolean nullReaders = readers == null;
            if (nullReaders) {
                AtomicReaderContext[] leaves =
                    indexReader().getTopReaderContext().leaves();
                readers = leaves.clone();
            }
            if (nullReaders || (requestCount.incrementAndGet() % 100) == 0) {
                Arrays.sort(readers, readerContextComparator);
            }
            return readers;
        }

        public IndexReader indexReader() {
            final MemoryPart old = this.old;
            IndexReader reader = indexReader.get();
            if (reader == null && old != null) {
                reader = old.indexReader();
            }
            return reader;
        }

        public void indexReader(final IndexReader indexReader) {
            readers = null;
            this.indexReader.set(indexReader);
        }

        public ConcurrentLinkedQueue<Document> defferedData() {
            return defferedData;
        }

        public String getLargestNode(final String parent) throws IOException {
            final String needle = parent + LARGEST_QUEUE_NODE;
            final IndexReader currentReader = indexReader();

            if (sortedSet.get() == null) {
                ConcurrentSkipListSet<String> newSet =
                    new ConcurrentSkipListSet<>();
                if (sortedSet.compareAndSet(null, newSet)) {
                    synchronized (newSet) {
                        newSet.addAll(liveData.keySet());
                        newSet.addAll(oldLiveData().keySet());
                        sortedKeysInitialized.set(true);
                        newSet.notifyAll();
                    }
                }
            }
            ConcurrentSkipListSet<String> sortedKeys = sortedSet.get();
            while (!sortedKeysInitialized.get()) {
                synchronized (sortedKeys) {
                    try {
                        sortedKeys.wait(1);
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
            final String floor = sortedKeys.floor(needle);
            if (floor != null && floor.startsWith(parent)) {
                return floor;
            }
            String largestNode = findLargestNode(parent, currentReader);
            if (largestNode != null) {
                return largestNode;
            }
            return null;
        }


        public long liveDataSize() {
            MemoryPart old = this.old;
            if (old == null) {
                return liveDataSize.get();
            } else {
                return liveDataSize.get() + old.liveDataSize();
            }
        }

        public void drainDeffered() throws IOException {
            for (Document doc: defferedData) {
                indexWriter.addDocument(doc);
            }
        }

        public void clear() {
            liveData.clear();
            hashMiss.clear();
        }

        public void addData(
            final String queuePath,
            final String path,
            final String hash,
            final long ctime,
            final byte[] data,
            final int weight)
            throws IOException
        {
            final Document doc = createDocument(
                queuePath,
                path,
                hash,
                ctime,
                data);
            if (indexReader.get() != null) {
                indexWriter.addDocument(doc);
            } else {
                defferedData.add(doc);
            }
            final Node d = new Node(data, path);
            liveData.put(path, d);
            if (hash != null) {
                hashMiss.remove(hash);
                liveData.put(hash, d);
            }
            liveDataSize.addAndGet(weight);
        }

        public void addOnCommitCallback(final Runnable callback) {
            onCommitCallbacks.add(callback);
        }

        public void afterCommit() {
            for (Runnable callback : onCommitCallbacks) {
                runCallback(callback);
            }
        }

        public MemoryPart old() {
            return old;
        }

        public void old(final MemoryPart old) {
            this.old = old;
        }
    }

    private static Set<String> storedFieldsInit() {
        HashSet<String> set = new HashSet<>();
        set.add("path");
        set.add("hash");
        set.add("time");
        set.add("data");
        set.add("queue");
        return set;
    }

    private static Set<String> indexedFieldsInit() {
        HashSet<String> set = new HashSet<>();
        set.add("path");
        set.add("hash");
        set.add("time");
        return set;
    }

    private static class Node {
        public final byte[] bytes;
        public final String path;

        public Node(
            final byte[] bytes,
            final String path)
        {
            this.bytes = bytes;
            this.path = path;
        }
    }

    public LuceneQueue(
        final LoadingCache<LuceneStorage.CacheKey, byte[]> diskCache,
        final long maxSegmentSize,
        final PrefixedLogger logger)
    {
        this.diskCache = diskCache;
        this.maxSegmentSize = maxSegmentSize;
        this.logger = logger;
    }

    public String name() {
        return dataDir.getName();
    }

    public boolean initialized() {
        return initialized;
    }

    public synchronized void init(
        final File storageDir,
        final String path,
        final SharedThreadPoolMergeScheduler mergeScheduler)
        throws IOException
    {
        if (initialized) {
            return;
        }
        dataDir = new File(storageDir, path);
        dataDir.mkdirs();
        dataDirectory = new NIOFSDirectory(dataDir) {
            @Override
            protected void fsync(String name) throws IOException {
            }

            @Override
            public IndexOutput createOutput(
                final String name)
                throws IOException
            {
                return WriteLimiter.INSTANCE.wrapIndexOutput(
                    (BufferedIndexOutput) super.createOutput(name));
            }
        };
        this.mergeScheduler = mergeScheduler;
        createIndexWriter();
        for (Map.Entry<String, String> entry: commitData.entrySet()) {
            final String key = entry.getKey();
            if (key.startsWith(LARGEST_NODE)) {
                final String queuePath = key.substring(LARGEST_NODE.length());
                final long value = Long.parseLong(entry.getValue());
                largestDiskNodes.put(queuePath, new AtomicLong(value));
                commitedLargestNodes.put(queuePath, value);
            }
        }
        currentMemoryPart.set(new MemoryPart(indexWriter));
        final IndexReader indexReader = indexReader(true);
        currentMemoryPart.get().indexReader(indexReader);
        initialized = true;
    }

    private IndexWriterConfig createIndexWriterConfig()
    {
        IndexWriterConfig cfg = new IndexWriterConfig( Version.LUCENE_40, new KeywordAnalyzer() );
        cfg.setRAMBufferSizeMB(RAM_BUFFER_SIZE_MB);
        mergePolicy = new TieredMergePolicy();
        mergePolicy.setUseCompoundFile( false );
        mergePolicy.setMaxMergeAtOnce( 50 );
        mergePolicy.setSegmentsPerTier( 30 );
        mergePolicy.setMaxMergedSegmentMB( maxSegmentSize / 1024 / 1024 );
        mergePolicy.setForceMergeDeletesPctAllowed( 5 );
        cfg.setMergePolicy(mergePolicy);
        cfg.setMergeScheduler(mergeScheduler);
        cfg.setMergedSegmentWarmer(new MergeNotifier());
        return cfg;
    }

    private void createIndexWriter() throws IOException
    {
        indexWriter = new IndexWriter(
                dataDirectory,
                createIndexWriterConfig(),
                storedFields,
                indexedFields);
        try {
            commitData = IndexReader.getCommitUserData(dataDirectory);
        } catch (org.apache.lucene.index.IndexNotFoundException ignore) {
            commitData = new HashMap<>();
        }
        mergePolicy.setExpungeAll(false);
        mergePolicy.setConvert(false);
        mergePolicy.setForceMergeDeletesPctAllowed(5.0);
        indexWriter.expungeDeletes(false);
    }

    private static Document createDocument(
        final String queuePath,
        final String path,
        final String hash,
        final long ctime,
        byte[] data)
    {
        Document doc = new Document();

        doc.add(
            new Field(
                "queue",
                queuePath,
                Field.Store.YES,
                Field.Index.NOT_ANALYZED_NO_NORMS));
        doc.add(
            new Field(
                "path",
                path,
                Field.Store.YES,
                Field.Index.NOT_ANALYZED_NO_NORMS));
        doc.add(
            new Field(
                "time",
                Long.toString(ctime / 1000),
                Field.Store.YES,
                Field.Index.NOT_ANALYZED_NO_NORMS));
        if (hash != null) {
            doc.add(
                new Field(
                    "hash",
                    hash,
                    Field.Store.YES,
                    Field.Index.NOT_ANALYZED_NO_NORMS));
        }
        doc.add(new Field("data", data));
        return doc;
    }

    private void runCallback(final Runnable callback) {
        try {
            callback.run();
        } catch (RuntimeException e) {
            logger.log(
                Level.SEVERE,
                "after commit IndexTask callback error",
                e);
        }
    }

    public int addData(
        final long position,
        final String queuePath,
        final String path,
        final String hash,
        final long ctime,
        final byte[] data,
        final WeighableRunnable callback)
        throws IOException
    {
        AtomicLong largestDiskNode = largestDiskNodes.get(queuePath);
        if (largestDiskNode == null) {
            AtomicLong newValue = new AtomicLong(NOT_INITIALIZED);
            largestDiskNode = largestDiskNodes.putIfAbsent(queuePath, newValue);
            if (largestDiskNode == null) {
                largestDiskNode = newValue;
            }
        }
        if (largestDiskNode.get() == NOT_INITIALIZED) {
            synchronized (largestDiskNode) {
                if (largestDiskNode.get() == NOT_INITIALIZED) {
                    String largestDiskNodePath = getLargestNode(queuePath);
                    if (largestDiskNodePath == null) {
                        largestDiskNode.set(-1L);
                    } else {
                        largestDiskNode.set(
                            Long.parseLong(
                                largestDiskNodePath.substring(
                                    largestDiskNodePath.length()
                                    - DataTree.QUEUE_NODE_DECIMAL_COUNT)));
                    }
                }
            }
        }
        indexLock.readLock().lock();
        try {
            final MemoryPart current = currentMemoryPart.get();
            final int weight;
            long largestNodeId = largestDiskNode.get();
            if (position > largestNodeId) {
                weight = dataSize(data, path, hash) + callback.weight() + 32;
                current.addData(queuePath, path, hash, ctime, data, weight);
                largestDiskNode.set(position);
                current.addOnCommitCallback(callback);
            } else {
                long commitedPosition =
                    commitedLargestNodes.getOrDefault(queuePath, -1L);
                logger.warning(
                    "Skipping message with position: " + position
                    + ", for shard: " + queuePath
                    + ", currentPosition: " + largestNodeId
                    + ", commitedPosition: " + commitedPosition);
                if (position > commitedPosition) {
                    weight = callback.weight() + 32;
                    current.liveDataSize.addAndGet(weight);
                    current.addOnCommitCallback(callback);
                } else {
                    weight = 0;
                    runCallback(callback);
                }
            }
            return weight;
        } finally {
            indexLock.readLock().unlock();
        }
    }

    private static int dataSize(
        final byte[] data,
        final String path,
        final String hash)
    {
        int size = DATA_OVERHEAD + data.length;
        size += path.length() << 1;
        if (hash != null) {
            size += hash.length() << 1;
        }
        return size;
    }

    public long size() {
        return currentMemoryPart.get().liveDataSize();
    }

    private Thread compareAndSetFlusher() {
        Thread current = Thread.currentThread();
        Thread flusher;
        while (true) {
            flusher = flushLock.get();
            boolean success = flushLock.compareAndSet(null, current);
            if (success) {
                return null;
            } else {
                if (flusher == flushLock.get()) {
                    return flusher;
                }
            }
        }
    }

    public void reopen() throws IOException {
        Thread currentFlusher = compareAndSetFlusher();
        if (currentFlusher == null) {
            try {
                doReopen();
            } finally {
                flushLock.set(null);
            }
        } else {
            logger.warning(
                "Flush or Reopen already in progress by thread: "
                + currentFlusher
                + ". No concurrent reopen allowed with live data size: "
                + currentMemoryPart.get().liveDataSize());
        }
    }

    private void doReopen() throws IOException {
        indexLock.writeLock().lock();
        final MemoryPart currentPart = currentMemoryPart.get();

        logger.info(
            "Reopening (lite) reader, dataSize: " + currentPart.liveDataSize());

        final IndexReader oldReader = currentPart.indexReader();
        boolean success = false;
        try {
            final IndexReader newIndexReader = indexReader(false);
            success = true;
            readLock.writeLock().lock();
            currentPart.indexReader(newIndexReader);
        } finally {
            if (success) {
                readLock.writeLock().unlock();
            }
            indexLock.writeLock().unlock();
        }
        if (success) {
            closeReader(oldReader);
        }
    }

    public boolean flush() throws IOException {
        Thread currentFlusher = compareAndSetFlusher();
        if (currentFlusher == null) {
            try {
                return doFlush();
            } finally {
                flushLock.set(null);
            }
        } else {
            logger.warning(
                "Flush or Reopen already in progress by thread: "
                + currentFlusher
                + ". No concurrent flush allowed with live data size: "
                + currentMemoryPart.get().liveDataSize());
            return false;
        }
    }

    private boolean doFlush() throws IOException {
        if (currentMemoryPart.get().liveDataSize() == 0
            && !segmentsChanged
            && indexWriter.getBufferedDeleteTermsSize() == 0)
        {
            logger.info("Skipping reopening reader: no new data");
            return false;
        }
        final MemoryPart newPart = new MemoryPart(indexWriter);
        final MemoryPart oldPart;
        final HashMap<String, Long> positionsSnapshot = new HashMap<>();
        indexLock.writeLock().lock();
        try {
            oldPart = currentMemoryPart.get();
            newPart.old(oldPart);
            currentMemoryPart.set(newPart);
            for (Map.Entry<String, AtomicLong> entry: largestDiskNodes.entrySet()) {
                final long position = entry.getValue().get();
                if (position != NOT_INITIALIZED) {
                    commitData.put(
                        LARGEST_NODE + entry.getKey(),
                        Long.toString(entry.getValue().get()));
                    positionsSnapshot.put(entry.getKey(), entry.getValue().get());
                }
            }
        } finally {
            indexLock.writeLock().unlock();
        }

        logger.info("Reopening reader, dataSize: " + oldPart.liveDataSize());
        logger.info("Deletes: " + indexWriter.getBufferedDeleteTermsSize());
        indexWriter.commit(commitData);
        final IndexReader newIndexReader = indexReader(true);

        commitedLargestNodes.putAll(positionsSnapshot);

        indexLock.writeLock().lock();
        readLock.writeLock().lock();
        try {
            newPart.indexReader(newIndexReader);
            newPart.drainDeffered();
            newPart.old(null);
        } finally {
            readLock.writeLock().unlock();
            indexLock.writeLock().unlock();
        }

        closeReader(oldPart.indexReader());
        oldPart.afterCommit();
        oldPart.clear();
        return true;
    }

    private boolean exist(
        final String field,
        final String key,
        final IndexReader indexReader)
        throws IOException
    {
        BytesRef term = null;

        AtomicReaderContext[] atomicReaders =
            indexReader.getTopReaderContext().leaves();
        BytesRef keyRef = new BytesRef(key);
        for (AtomicReaderContext arc : atomicReaders) {
            IndexReader reader = arc.reader;

            Terms terms = reader.fields().terms(field);
            if (terms == null) {
                logger.warning("findDocId.terms = null");
                continue;
            }

            TermsEnum te = terms.getThreadTermsEnum(true);
            if (te == null) {
                continue;
            }
            try {
                DocsEnum de = null;
                if (te.seekExact(keyRef, true) == false) {
                    continue;
                }
                term = te.term();
                if (term == null) {
                    logger.warning("findDocId: term.text() is null after "
                        + "TermsEnum.SeekStatus.EXACT status for path: " + key);
                    continue;
                }
                de = te.docs(null, de);
                int docId;
                if((docId = de.nextDoc()) != DocIdSetIterator.NO_MORE_DOCS) {
                    return true;
                }
            } finally {
                te.close();
            }
        }
        return false;
    }

    private Comparator<AtomicReaderContext> readerContextComparator(
        final String field)
    {
        Comparator<AtomicReaderContext> comp = contextsComparators.get(field);
        if (comp == null) {
            comp = new ReaderContextComparator(field);
            contextsComparators.put(field, comp);
        }
        return comp;
    }

    private Document findDoc(
        final String field,
        final String key,
        final AtomicReaderContext[] readers)
//        final IndexReader indexReader)
        throws IOException
    {
        BytesRef term = null;

//        AtomicReaderContext[] atomicReaders =
//            indexReader.getTopReaderContext().leaves();
//        Arrays.sort(atomicReaders, readerContextComparator(field));
        for (AtomicReaderContext arc : readers) {
            IndexReader reader = arc.reader;
//          Bits liveDocs = reader.getLiveDocs();
            Bits liveDocs = null; //allow to get deleted but not expunged yet docs

            Terms terms = reader.fields().terms(field);
            if (terms == null) {
                logger.warning("findDocId.terms = null");
                continue;
            }

            TermsEnum te = terms.getThreadTermsEnum(true);
            if (te == null) {
                continue;
            }
            try {
                DocsEnum de = null;

                if (te.seekExact( new BytesRef( key ), false ) == false) {
                    continue;
                }
                term = te.term();
                if (term == null) {
                    logger.warning("findDocId: term.text() is null after "
                        + "TermsEnum.SeekStatus.EXACT status for path: " + key);
                    continue;
                }

                hitCounter(arc, field.intern()).incrementAndGet();
                de = te.docs(liveDocs, de);
                int count = 0;
                int docId;
                int prevDocId = -1;
                while((docId = de.nextDoc()) != DocIdSetIterator.NO_MORE_DOCS) {
                    prevDocId = docId;
                    count++;
                }
                if (count == 0) {
                    continue;
                }
                if (count > 1) {
                    logger.warning(
                        "findDocId: found <" + count + "> docId for key: " + key
                            + ">. DataNode was indexed more than once!");
                }
                return reader.document(prevDocId);//prevDocId + arc.docBase;
            } finally {
                te.close();
            }
        }
//      System.err.println("findDocId.needle: <" + key +"> failed" );
        return null;
    }

    private String findNextNode(final String parent, final String path,
        final IndexReader reader, Bits liveDocs) throws IOException
    {
        BytesRef term = null;
        String field = "path";
        String minPath = path;
        Fields fields = reader.fields();
        Terms terms = fields.terms("path");
        BytesRef key = new BytesRef(path);
        BytesRef parentKey = new BytesRef(parent);
        if (terms == null) {
            logger.warning("findNextNode<"+reader+">.terms = null on " + path);
            return null;
        }

        TermsEnum te = terms.getThreadTermsEnum(true);
        if (te == null) {
            return null;
        }
        try {
            DocsEnum de = null;

            TermsEnum.SeekStatus seekStatus = te.seek(key, true);
            if (seekStatus != TermsEnum.SeekStatus.NOT_FOUND &&
                seekStatus != TermsEnum.SeekStatus.FOUND )
            {
                return null;
            }

            for(term = te.term(); term != null; term = te.next()) {
                if (term.equals(key)) {
                    continue;
                }
                if (!term.startsWith(parentKey)) {
                    return null;
                }

                de = te.docs( liveDocs, de );
                int count = 0;
                int docId;
                int prevDocId = -1;
                while ((docId = de.nextDoc())
                    != DocIdSetIterator.NO_MORE_DOCS)
                {
                    prevDocId = docId;
                    count++;
                }
                if (count == 0) {
                    continue;
                }
                if (count > 0) {
                    return term.utf8ToString();
                }
            }
        } finally {
            te.close();
        }
        return null;
    }

    private String findNextNode(
        final String parent,
        final String path,
        final IndexReader indexReader)
        throws IOException
    {
        String minPath = null;
        AtomicReaderContext[] atomicReaders =
            indexReader.getTopReaderContext().leaves();
        for (AtomicReaderContext arc : atomicReaders) {
            IndexReader reader = arc.reader;
            Bits liveDocs = reader.getDeletedDocs();
            String nextPath = findNextNode(parent, path, reader, liveDocs);
            if (nextPath != null) {
                if (minPath == null) {
                    minPath = nextPath;
                } else if (minPath.compareTo(nextPath) > 0) {
                    minPath = nextPath;
                }
            }
        }
        return minPath;
    }

    private String findLargestNode(
        final String parent,
        final IndexReader reader,
        final Bits deletes) throws IOException
    {
        BytesRef term = null;
        String field = "path";
        Fields fields = reader.fields();
        Terms terms = fields.terms("path");
        if (terms == null) {
            logger.warning("findNextNode<" + reader + ">.terms = null on " + parent);
            return null;
        }

        TermsEnum te = terms.reverseIterator();
        if (te == null) {
            return null;
        }

        DocsEnum de = null;

        BytesRef parentKey = new BytesRef(parent);
        BytesRef key = new BytesRef(parent + LARGEST_QUEUE_NODE);
        TermsEnum.SeekStatus seekStatus = te.seek(key, true);
        if (seekStatus != TermsEnum.SeekStatus.NOT_FOUND &&
            seekStatus != TermsEnum.SeekStatus.FOUND )
        {
            return null;
        }

        for(term = te.term(); term != null; term = te.next()) {
            if (!term.startsWith(parentKey)) {
                break;
            }

            de = te.docs(deletes, de);
            int count = 0;
            int docId;
            if ((docId = de.nextDoc()) != DocIdSetIterator.NO_MORE_DOCS) {
                return term.utf8ToString();
            }
        }
        return null;
    }

    private String findLargestNode(
        final String parent,
        final IndexReader indexReader)
        throws IOException
    {
        String maxPath = null;
        AtomicReaderContext[] atomicReaders =
            indexReader.getTopReaderContext().leaves();
        for (AtomicReaderContext arc : atomicReaders) {
            IndexReader reader = arc.reader;
            Bits deletes = reader.getDeletedDocs();
            String nextPath = findLargestNode(parent, reader, deletes);
//            LOG.info("findLargestNode(" + parent + "): " + nextPath);
            if (nextPath != null) {
                if (maxPath == null) {
                    maxPath = nextPath;
                } else if (maxPath.compareTo(nextPath) < 0) {
                    maxPath = nextPath;
                }
            }
        }
        return maxPath;
    }

    public byte[] readNodeFromIndex(
        final String path,
        final AtomicReaderContext[] readers)
        throws KeeperException.SystemErrorException
    {
        try
        {
//            ensureOpen();
            Document doc = findDoc("path", path, readers);
            if( doc == null ) return null;// throw new KeeperException.NoNodeException();
            return doc.getBinaryValue("data");
        }
        catch( IOException e )
        {
            throw new KeeperException.SystemErrorException();
        }
    }

    private byte[] readNodeFromIndexByHash( String hash, AtomicReaderContext[] readers ) throws KeeperException.SystemErrorException
    {
        try
        {
//            ensureOpen();
            Document doc = findDoc( "hash", hash, readers );
            if( doc == null ) return null;// throw new KeeperException.NoNodeException();
            return doc.getBinaryValue("data");
        }
        catch( IOException e )
        {
            throw new KeeperException.SystemErrorException();
        }
    }


    private boolean exist(final String path)
        throws IOException
    {
        readLock.readLock().lock();
        try {
            final MemoryPart currentPart = currentMemoryPart.get();
            final NonBlockingHashMap<String, Node> liveData =
                currentPart.liveData();
            final Map<String, Node> oldLiveData =
                currentPart.oldLiveData();
            final IndexReader currentReader = currentPart.indexReader();
            Node data = liveData.get(path);
            if (data != null) {
                return true;
            }

            data = oldLiveData.get(path);
            if (data != null) {
                return true;
            }

            byte[] byteData = diskCache.getIfPresent(
                new LuceneStorage.CacheKey(this, path, currentPart.readers()));
            if (byteData != null) {
                return true;
            }
            return exist("path", path, currentReader);
        } finally {
            readLock.readLock().unlock();
        }
    }

    private byte[] readNodeDirect( String path ) throws KeeperException.SystemErrorException
    {
        readLock.readLock().lock();
        try {
            final MemoryPart currentPart = currentMemoryPart.get();
            final NonBlockingHashMap<String, Node> liveData =
                currentPart.liveData();
            final Map<String, Node> oldLiveData =
                currentPart.oldLiveData();
            final IndexReader currentReader = currentPart.indexReader();
            Node data = liveData.get( path );
            if( data != null ) return data.bytes;

            data = oldLiveData.get(path);
            if (data != null) {
                return data.bytes;
            }

            if (path.startsWith("hash/")) {
                return readNodeFromIndexByHash(path, currentPart.readers());
            } else {
                try {
                    return diskCache.get(
                        new LuceneStorage.CacheKey(this, path, currentPart.readers()));
                } catch (ExecutionException e) {
                    Throwable cause = e.getCause();
                    if (cause instanceof KeeperException.NoNodeException) {
                        return null;
                    } else if (
                        cause instanceof KeeperException.SystemErrorException)
                    {
                        throw (KeeperException.SystemErrorException) cause;
                    } else {
                        logger.log(Level.SEVERE, "Unhandled exception", cause);
                        throw new KeeperException.SystemErrorException();
                    }
                }
            }
        } finally {
            readLock.readLock().unlock();
        }
    }

    public byte[] readData(final String path)
        throws KeeperException.SystemErrorException
    {
        return readNodeDirect(path);
    }

    public String getNextNode(final String parent, final String path)
        throws KeeperException.SystemErrorException
    {
        readLock.readLock().lock();
        try {
            final MemoryPart currentPart = currentMemoryPart.get();
            final NonBlockingHashMap<String, Node> liveData =
                currentPart.liveData();
            final Map<String, Node> oldLiveData =
                currentPart.oldLiveData();
            final IndexReader currentReader = currentPart.indexReader();
            try {
                String nextNode = findNextNode(parent, path, currentReader);
                if (nextNode != null) {
                    return nextNode;
                }
            } catch (IOException e) {
                throw new KeeperException.SystemErrorException();
            }

            //SLOW MODE ON
            TreeSet<String> sortedSet = new TreeSet<String>(liveData.keySet());
            sortedSet.addAll(oldLiveData.keySet());
            //SLOW MODE OFF
            Set<String> tailSet = sortedSet.tailSet(path);
            Iterator<String> iter = tailSet.iterator();
            if (iter.hasNext()) {
                String next = iter.next();
                if (next.startsWith(parent)) {
                    return next;
                }
            }
            return null;
        } finally {
            readLock.readLock().unlock();
        }
    }

    public String getLargestNode(final String parent) throws IOException {
        readLock.readLock().lock();
        try {
            final MemoryPart currentPart = currentMemoryPart.get();
            return currentPart.getLargestNode(parent);
        } finally {
            readLock.readLock().unlock();
        }
    }

    public String getPathByHash(final String hash)
        throws KeeperException.SystemErrorException
    {
        readLock.readLock().lock();
        try {
            final MemoryPart currentPart = currentMemoryPart.get();
            final NonBlockingHashMap<String, Node> liveData =
                currentPart.liveData();
            final Map<String, Node> oldLiveData =
                currentPart.oldLiveData();
            final IndexReader currentReader = currentPart.indexReader();
            Node data = liveData.get(hash);
            if (data != null) {
                return data.path;
            }

            data = oldLiveData.get(hash);
            if (data != null) {
                return data.path;
            }
            if (currentPart.hashMiss.get(hash) != null) {
                return null;
            }
            try {
                Document doc = findDoc("hash", hash, currentPart.readers());
                if (doc == null) {
                    currentPart.hashMiss.put(hash, hash);
                    data = liveData.get(hash);
                    if (data != null) {
                        currentPart.hashMiss.remove(hash);
                        return data.path;
                    }
                    return null;// throw new KeeperException.NoNodeException();
                }
                return doc.getField("path").stringValue();
            } catch (IOException e) {
                throw new KeeperException.SystemErrorException();
            }
        } finally {
            readLock.readLock().unlock();
        }
    }

    public long storageSize() throws IOException {
        long size = 0;
        if (!initialized) {
            return 0;
        }
        String[] files = dataDirectory.listAll();
        for (String file : files) {
            try {
                size += dataDirectory.fileLength(file);
            } catch (java.io.FileNotFoundException e) {
                logger.log(
                    Level.WARNING,
                    "Error on file <" + file + '>',
                    e);
            }
        }
        return size;
    }

    private static AtomicInteger hitCounter(
        final AtomicReaderContext ctx,
        final String field)
    {
        Object userCtx = ctx.userContext();
        if (userCtx == null) {
            userCtx = new AtomicInteger(0);
            ctx.userContext(userCtx);
        }
        return (AtomicInteger) userCtx;
/*        Map<String, AtomicInteger> hitCountMap;
        if (userCtx == null) {
            hitCountMap = new IdentityHashMap<String, AtomicInteger>();
            ctx.userContext(hitCountMap);
        } else {
            hitCountMap = (Map<String, AtomicInteger>) userCtx;
        }
        AtomicInteger i = hitCountMap.get(field);
        if (i == null) {
            i = hitCountMap.computeIfAbsent(
                field,
                k -> new AtomicInteger(0));
        }
*/
//        return i;
    }

    public IndexReader indexReader(final boolean applyDeletes)
        throws IOException
    {
        final IndexReader reader = IndexReader.open(indexWriter, applyDeletes);
        reader.incRef();
        return reader;
    }

    public void removeTerm(final Term term) throws IOException {
        indexWriter.deleteDocuments(term);
    }

    public void closeReader(final IndexReader reader)
        throws IOException
    {
        final ReaderFinished finishChecker = new ReaderFinished(reader);
        reader.addReaderFinishedListener(finishChecker);
        while (!finishChecker.finished()) {
            reader.decRef();
        }
    }

    public void afterTruncate() throws IOException {
        logger.warning("Running light expunge");
        mergePolicy.setExpungeAll(false);
        mergePolicy.setConvert(false);
        mergePolicy.setForceMergeDeletesPctAllowed(5.0);
        indexWriter.maybeMerge();
        indexWriter.expungeDeletes(true);
        logger.warning("Light expunge finished");
        flush();
    }

    public void expunge() throws IOException {
        mergePolicy.setForceMergeDeletesPctAllowed(0);
        logger.warning("Running hard expunge");
        indexWriter.expungeDeletes(true);
        logger.warning("Hard expunge finished");
        mergePolicy.setForceMergeDeletesPctAllowed(5.0);
        flush();
    }

    private class MergeNotifier extends IndexReaderWarmer {

        @Override
        public void indexesAdded() {
            segmentsChanged = true;
        }

        @Override
        public void warm(IndexReader reader) {
        }

        @Override
        public void unwarm(IndexReader reader) {
        }

        @Override
        public void mergeSuccess() {
            logger.info(
                "Background merge has been finished. flushing to reopen new segments");
            segmentsChanged = true;
            try {
                reopen();
            } catch (IOException e) {
                logger.log(
                    Level.WARNING,
                    "IOException trying to flush shard after merge",
                    e);
            }
        }
    }

    private static class ReaderContextComparator
        implements Comparator<AtomicReaderContext>
    {
        private final String field;

        ReaderContextComparator(final String field) {
            this.field = field.intern();
        }

        @Override
        public int compare(
            final AtomicReaderContext a,
            final AtomicReaderContext b)
        {
            return Integer.compare(
                hitCounter(b, field).get(),
                hitCounter(a, field).get());
        }
    }
}
