package org.apache.zookeeper.server.persistence;

import com.google.common.cache.LoadingCache;
import com.google.common.cache.Weigher;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

import org.apache.zookeeper.KeeperException;
import org.apache.lucene.index.DocsEnum;
import org.apache.lucene.index.IndexReader.AtomicReaderContext;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.MultiFields;
import org.apache.lucene.index.SharedThreadPoolMergeScheduler;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.index.codecs.CodecProvider;
import org.apache.lucene.index.codecs.yandex.YandexCodec;
import org.apache.lucene.index.codecs.yandex2.Yandex2Codec;
import org.apache.lucene.store.Compressor;
import org.apache.lucene.store.LZ4Compressor;
import org.apache.lucene.store.ZstdCompressor;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.PriorityQueue;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.concurrent.WeighableRunnable;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.util.Compress;
import ru.yandex.parser.string.BooleanParser;

public class LuceneStorage
{
    //private static final Logger LOG = LoggerFactory.getLogger(LuceneStorage.class);
    private static final int CACHE_ENTRY_WEIGHT_OVERHEAD = 80;
    private static final int AVERAGE_OBJECT_SIZE = 20;
    private static final int CACHE_CONCURRENCY = 32;
    private static final int CACHE_MAX_WEIGHT = maxCacheWeight();
    private static final int INDEXING_THREAD_COUNT = 4;
    private static final int MERGE_THREAD_COUNT = 2;
    private static final int QUEUES_COUNT =
        Integer.parseInt(System.getProperty("ru.yandex.lucene-storage.queues-count", "100"));
    private static final long OVERRIDDEN_MAX_SEGMENT_SIZE =
        Long.parseLong(System.getProperty("ru.yandex.lucene-storage.max-segment-size", "0"));
    private static final AtomicLong LIVE_DATA_SIZE = new AtomicLong(0);
    private static final long MAX_LIVE_DATA_SIZE = MAX_LIVE_DATA_SIZE();
    private static final boolean ZOO_HASH_BLOOM_FILTER =
        BooleanParser.INSTANCE.apply(
            System.getProperty(
                "ru.yandex.lucene-storage.zoo-hash-bloom-filter",
                "true"));
    private static final boolean IN_MEMORY_FIELDS_INDEX =
        BooleanParser.INSTANCE.apply(
            System.getProperty(
                "ru.yandex.lucene-storage.in-memory-fields-index",
                "true"));
    private static final String LUCENE_CODEC =
        System.getProperty("ru.yandex.lucene-storage.codec", "Yandex2_zstd");
    private static final int LUCENE_CODEC_COMPRESSION_LEVEL =
        Integer.parseInt(
            System.getProperty(
                "ru.yandex.lucene-storage.codec-compression-level",
                "1"));
    private static final String LUCENE_GROUP_FIELDS =
        System.getProperty("ru.yandex.lucene-storage.group-fields", "queue");
    private static final boolean INDEX_WRITER_INFO_STREAM =
        BooleanParser.INSTANCE.apply(
            System.getProperty(
                "ru.yandex.lucene-storage.index-writer-info-stream",
                "false"));
    private static final HashMap<File, LuceneStorage> instances =
        new HashMap<File, LuceneStorage>();

    static {
        if (INDEX_WRITER_INFO_STREAM) {
            IndexWriter.setDefaultInfoStream(System.err);
        }
    }

private final Map<Integer, LuceneQueue> queues = new ConcurrentHashMap<>();
private final long maxSegmentSize;
private final LoadingCache<CacheKey, byte[]> diskCache;
private final SharedThreadPoolMergeScheduler concurrentScheduler;
private final PrefixedLogger logger;
private final boolean useCompoundIndex;
private final int indexThreads;
private File storageDir;
private IndexingThread[] indexingThreads;
private Thread indexFlusher;// = new IndexFlusher
private Thread storageCleaner;
private final long maxQueueStorageSize;
private final long queueStorageSizeDelta;

    private static int maxCacheWeight() {
        String size = System.getProperty(
            "ru.yandex.lucene-storage.max-cache-weight",
            "536870912");
        return Integer.parseInt(size);
    }

    private static long MAX_LIVE_DATA_SIZE() {
        String size = System.getProperty(
            "ru.yandex.lucene-storage.max-live-data-size",
            "400000000");
        return Long.parseLong(size);
    }

    private static boolean useCompoundIndex() {
        return Boolean.getBoolean(
            "ru.yandex.lucene-storage.use-compound-index");
    }

    private static int indexingThreadCount() {
        return Integer.getInteger(
            "ru.yandex.lucene-storage.index-threads",
            INDEXING_THREAD_COUNT);
    }

    private static int mergeThreadCount() {
        return Integer.getInteger(
            "ru.yandex.lucene-storage.merge-threads",
            MERGE_THREAD_COUNT);
    }

    public static LuceneStorage getInstance(
        final File path,
        final long maxQueueStorageSize,
        final PrefixedLogger logger)
        throws IOException
    {
        synchronized (instances) {
            System.err.println("LuceneStorage.getInstacne: "
                + path + ": " + instances.toString());
            LuceneStorage storage = instances.get(path);
            if (storage == null) {
                storage = new LuceneStorage(path, maxQueueStorageSize, logger);
                instances.put(path, storage);
            }
            return storage;
        }
    }

    private LuceneStorage(
        final File storageDir,
        final long maxQueueStorageSize,
        final PrefixedLogger logger)
        throws IOException
    {
        this.storageDir = storageDir;
        this.maxQueueStorageSize = maxQueueStorageSize;
        this.logger = logger;
        this.useCompoundIndex = useCompoundIndex();
        Compress.fadviseEnabled(false);
        queueStorageSizeDelta =
            maxQueueStorageSize / (long)Math.log(maxQueueStorageSize);
        if (OVERRIDDEN_MAX_SEGMENT_SIZE == 0L) {
            if (useCompoundIndex) {
                maxSegmentSize = queueStorageSizeDelta / 2;
            } else {
                maxSegmentSize = queueStorageSizeDelta / 2 / QUEUES_COUNT;
            }
        } else {
            maxSegmentSize = OVERRIDDEN_MAX_SEGMENT_SIZE;
        }
        logger.info("LuceneStorage.maxQueueStorageSize: "
            + maxQueueStorageSize + ", queueStorageSizeDelta: "
            + queueStorageSizeDelta);
        indexFlusher = new IndexFlusher();
        indexFlusher.start();
        storageCleaner = new StorageCleaner();
        storageCleaner.start();
        indexThreads = indexingThreadCount();
        indexingThreads = new IndexingThread[indexThreads];
        for (int i = 0; i < indexThreads; i++) {
            indexingThreads[i] = new IndexingThread(i);
            indexingThreads[i].start();
        }
        diskCache = CacheBuilder.newBuilder()
            .concurrencyLevel(CACHE_CONCURRENCY)
            .maximumWeight(CACHE_MAX_WEIGHT)
            .weigher(dataWeigher)
            .recordStats()
            .build(dataLoader);
        concurrentScheduler = new SharedThreadPoolMergeScheduler(
            mergeThreadCount());
        concurrentScheduler.setMergeThreadPriority(
            Thread.MIN_PRIORITY,
            Compress.IOPRIO_CLASS_IDLE,
            Compress.IOPRIO_HIGH);
        CodecProvider cp = CodecProvider.getDefault();
        HashSet<String> bloom = new HashSet<>();
        bloom.add("path");
        if (ZOO_HASH_BLOOM_FILTER) {
            bloom.add("hash");
        }
        Compressor[] compressors = new Compressor[] {
            new LZ4Compressor(LUCENE_CODEC_COMPRESSION_LEVEL),
            new ZstdCompressor(LUCENE_CODEC_COMPRESSION_LEVEL)
        };
        Set<String> groupFields =
            CollectionCompactor.compact(
                new HashSet<>(Arrays.asList(LUCENE_GROUP_FIELDS.split(","))));
        for (Compressor compressor: compressors) {
            cp.registerIfAbsent(
                new YandexCodec(
                    compressor,
                    6 * 1024,
                    2 * 1024,
                    4 * 1024,
                    groupFields,
                    bloom,
                    null)
                    .setFieldIndexReadBufferSize(
                        YandexCodec.DEFAULT_FIELD_INDEX_READ_BUFFER_SIZE)
                    .useInMemoryFieldsIndex(IN_MEMORY_FIELDS_INDEX)
                    .useStandardFieldsWriter(false));
            cp.registerIfAbsent(
                new Yandex2Codec(
                    compressor,
                    6 * 1024,
                    2 * 1024,
                    4 * 1024,
                    groupFields,
                    bloom,
                    null)
                    .setFieldIndexReadBufferSize(
                        YandexCodec.DEFAULT_FIELD_INDEX_READ_BUFFER_SIZE)
                    .useInMemoryFieldsIndex(IN_MEMORY_FIELDS_INDEX)
                    .useStandardFieldsWriter(false));
        }
        cp.setDefaultFieldCodec(LUCENE_CODEC);
    }

    public String cacheStats() {
        return diskCache.stats().toString();
    }

    private static final int pathToCode( String path )
    {
        return Math.abs(path.hashCode() % QUEUES_COUNT);
    }

    private LuceneQueue getCompoundQueue(final String path) throws IOException {
        final int code = QUEUES_COUNT;
        LuceneQueue queue = queues.get(code);
        if (queue == null) {
            LuceneQueue newQueue =
                new LuceneQueue(
                    diskCache,
                    maxSegmentSize,
                    logger.addPrefix("LuceneQueue<compound>"));
            queue = queues.putIfAbsent(code, newQueue);
            if (queue == null) {
                queue = newQueue;
            }
        }
        if (!queue.initialized()) {
            queue.init(
                storageDir,
                "compound",
                concurrentScheduler);
        }
        return queue;
    }

    private LuceneQueue getShardedQueue(final String path) throws IOException {
        final int code = pathToCode(path);
        LuceneQueue queue = queues.get(code);
        if (queue == null) {
            LuceneQueue newQueue =
                new LuceneQueue(
                    diskCache,
                    maxSegmentSize,
                    logger.addPrefix("LuceneQueue<y" + code + '>'));
            queue = queues.putIfAbsent(code, newQueue);
            if (queue == null) {
                queue = newQueue;
            }
        }
        if (!queue.initialized()) {
            queue.init(
                storageDir,
                'y' + Integer.toString(code),
                concurrentScheduler);
        }
        return queue;
    }

    public LuceneQueue getQueue(final String path) throws IOException {
        if (useCompoundIndex) {
            return getCompoundQueue(path);
        } else {
            return getShardedQueue(path);
        }
    }

    public LuceneQueue getOldQueue(final String path) throws IOException {
        if (useCompoundIndex) {
            return getShardedQueue(path);
        } else {
            return getCompoundQueue(path);
        }
    }

    public void commit() throws IOException {
        for (Map.Entry<Integer, LuceneQueue> entry: queues.entrySet()) {
            LuceneQueue queue = entry.getValue();
            queue.flush();
        }
    }

    public void addDataInternal(
        final long position,
        final String queuePath,
        final String path,
        final String hash,
        final long ctime,
        final byte[] data,
        final WeighableRunnable callback)
        throws KeeperException
    {
        try {
            LuceneQueue queue = getQueue(queuePath);
            final int weight = queue.addData(
                position,
                queuePath,
                path,
                hash,
                ctime,
                data,
                callback);
            synchronized (LIVE_DATA_SIZE) {
                LIVE_DATA_SIZE.addAndGet(weight);
            }
        } catch (IOException e) {
            throw new KeeperException.SystemErrorException(e);
        }
    }

    public Future<Void> addData(
        final String queuePath,
        final String path,
        final String hash,
        final long position,
        final long ctime,
        final byte[] data,
        final WeighableRunnable callback)
        throws KeeperException
    {
        //to avoid DataTree.deleteNode race conditions (uneven deletions)
        //we are sharding tasks manualy here by queuePath
        try {
            IndexingThread thread =
                indexingThreads[
                    Math.abs(queuePath.hashCode() % indexThreads)];
            FutureTask<Void> ft =
                new FutureTask<Void>(
                    new IndexTask(
                        position,
                        queuePath,
                        path,
                        hash,
                        ctime,
                        data,
                        callback));
            thread.queue.put(ft);
            return ft;
        } catch(Exception e) {
            e.printStackTrace();
            throw new KeeperException.SystemErrorException(e);
        }
    }

    public byte[] getDataDirect(
        final String queuePath,
        final String path)
        throws KeeperException.SystemErrorException
    {
        try {
            LuceneQueue queue = getQueue(queuePath);
            byte[] data = queue.readData(path);
            if (data == null) {
                LuceneQueue vQueue = getOldQueue(queuePath);
                data = vQueue.readData(path);
            }
            return data;
        } catch (IOException e) {
            throw new KeeperException.SystemErrorException();
        }
    }

    public byte[] getData(
        final String queuePath,
        final String path)
        throws KeeperException.SystemErrorException
    {
        return getDataDirect(queuePath, path);
    }

    public String getNextNode(
        final String queuePath,
        final String path)
        throws KeeperException.SystemErrorException
    {
        try {
            LuceneQueue vQueue = getOldQueue(queuePath);
            String nextNode = vQueue.getNextNode(queuePath, path);
            if (nextNode == null) {
                LuceneQueue queue = getQueue(queuePath);
                nextNode = queue.getNextNode(queuePath, path);
            }
            return nextNode;
        } catch (IOException e) {
            throw new KeeperException.SystemErrorException();
        }
    }
/*
    public String getLargestNode(final String queuePath) throws IOException {
        LuceneQueue queue = getQueue(queuePath);
        return queue.getLargestNode(queuePath);
    }
*/
    public String getPathByHash(
        final String queuePath,
        final String hash)
        throws KeeperException.SystemErrorException
    {
        try {
            LuceneQueue queue = getQueue(queuePath);
            String path = queue.getPathByHash(hash);
            if (path == null) {
                LuceneQueue vQueue = getOldQueue(queuePath);
                path = vQueue.getPathByHash(hash);
            }
            return path;
        } catch (IOException e) {
            throw new KeeperException.SystemErrorException();
        }
    }

    private class IndexFlusher extends Thread {
        public IndexFlusher() {
            super("IndexFlusher");
        }

        @Override
        public void run() {
            while (true) {
                try {
                    int iter = 0;
                    while (LIVE_DATA_SIZE.get() < MAX_LIVE_DATA_SIZE) {
                        Thread.sleep(1000L);
                        if ((++iter % 60) == 0) {
                            logger.info(
                                "IndexFlusher.LIVE_DATA_SIZE = "
                                + LIVE_DATA_SIZE.get()
                                + " at wait iteration #" + iter);
                        }
                    }

                    long liveDataSize = 0L;
                    synchronized (LIVE_DATA_SIZE) {
                        for (LuceneQueue queue: queues.values()) {
                            liveDataSize += queue.size();
                        }
                        LIVE_DATA_SIZE.set(liveDataSize);
                    }

                    logger.info(
                        "IndexFlusher.LIVE_DATA_SIZE = " + liveDataSize);
                    long maxQueueSize = -1;
                    LuceneQueue maxQueue = null;
                    for (LuceneQueue queue: queues.values()) {
                        long size = queue.size();
                        if (size > maxQueueSize) {
                            maxQueue = queue;
                            maxQueueSize = size;
                        }
                    }
                    if (maxQueue != null && maxQueue.flush()) {
                        liveDataSize = 0L;
                        synchronized (LIVE_DATA_SIZE) {
                            for (LuceneQueue queue: queues.values()) {
                                liveDataSize += queue.size();
                            }
                            LIVE_DATA_SIZE.set(liveDataSize);
                        }
                        logger.info(
                            "IndexFlusher.LIVE_DATA_SIZE = " + liveDataSize
                            + " after LuceneQueue<" + maxQueue.name()
                            + "> flush, queue size changed " + maxQueueSize
                            + " -> " + maxQueue.size());
                    }
                    Thread.sleep(100L);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private interface ComparableQueue {
        void init() throws IOException;

        int numDeletedDocs();

        DocsEnum docsIterator() throws IOException;

        BytesRef currentTerm();

        int averageMessageSize();

        LuceneQueue queue();

        void removeCurrent() throws IOException;

        void next() throws IOException;

        String name();

        void flush() throws IOException;

        void afterTruncate() throws IOException;

        void expunge() throws IOException;

        void close();
    }

    private static final class YandexComparableQueue
        implements ComparableQueue
    {
        private final LuceneQueue queue;
        private IndexReader indexReader = null;
        private TermsEnum termsIterator = null;
        private DocsEnum docsIterator = null;
        private Bits deletedDocs = null;
        private BytesRef currentTerm = null;
        private int averageMessageSize;
        private final PrefixedLogger logger;

        YandexComparableQueue(
            final PrefixedLogger logger,
            final LuceneQueue queue)
        {
            this.logger = logger;
            this.queue = queue;
        }

        @Override
        public void init() throws IOException {
            indexReader = queue.indexReader(true);
            Terms terms = MultiFields.getTerms(indexReader, "time");
            if (terms != null) {
                averageMessageSize =
                    (int) Math.ceil(queue.storageSize() / indexReader.maxDoc());
                deletedDocs = MultiFields.getDeletedDocs(indexReader);
                termsIterator = terms.iterator(false);
                next();
            }
        }

        @Override
        public int numDeletedDocs() {
            if (indexReader == null) {
                return 0;
            } else {
                return indexReader.numDeletedDocs();
            }
        }

        @Override
        public DocsEnum docsIterator() throws IOException {
            docsIterator = termsIterator.docs(deletedDocs, docsIterator);
            return docsIterator;
        }

        @Override
        public BytesRef currentTerm() {
            return currentTerm;
        }

        @Override
        public int averageMessageSize() {
            return averageMessageSize;
        }

        @Override
        public void removeCurrent() throws IOException {
            queue.removeTerm(new Term("time", new BytesRef(currentTerm)));
        }

        @Override
        public void next() throws IOException {
            currentTerm = termsIterator.next();
        }

        @Override
        public LuceneQueue queue() {
            return queue;
        }

        @Override
        public String name() {
            return queue.name();
        }

        @Override
        public void flush() throws IOException {
            close();
            queue.flush();
        }

        @Override
        public void afterTruncate() throws IOException {
            queue.afterTruncate();
        }

        @Override
        public void expunge() throws IOException {
            queue.expunge();
        }

        @Override
        public void close() {
            try {
                if (indexReader != null) {
                    queue.closeReader(indexReader);
                    indexReader = null;
                }
            } catch (Exception e) {
                logger.log(Level.SEVERE, "Exception closing IndexReader after truncate", e);
            }
        }
    }

    private static final class TimePriorityQueue
        extends PriorityQueue<ComparableQueue>
    {
        TimePriorityQueue(final int size) {
            super(size);
        }

        @Override
        public boolean lessThan(
            final ComparableQueue a,
            final ComparableQueue b)
        {
            if (a.currentTerm() == null && b.currentTerm() == null) {
                return false;
            }
            if (a.currentTerm() == null) {
                //NULL is bigger
                return false;
            }
            if (b.currentTerm() == null) {
                return true;
            }
            return a.currentTerm().compareTo(b.currentTerm()) < 0;
        }
    }

    private class StorageCleaner extends Thread {
        StorageCleaner() {
            super("StorageCleaner");
        }

        private IdentityHashMap<ComparableQueue, int[]> truncate(
            final TimePriorityQueue pq,
            final long toRemove)
            throws IOException
        {
            final IdentityHashMap<ComparableQueue, int[]> removesCount =
                new IdentityHashMap<>(pq.size() << 1);
            long removed = 0;
            for (
                ComparableQueue cq = pq.top();
                cq.currentTerm() != null && removed < toRemove;
                cq.next(), pq.updateTop(), cq = pq.top())
            {
                final DocsEnum de = cq.docsIterator();
                int count = 0;
                while (de.nextDoc() != DocIdSetIterator.NO_MORE_DOCS) {
                    count++;
                }
                if (count > 0) {
                    int[] queueCount = removesCount.get(cq);
                    if (queueCount == null) {
                        queueCount = new int[2];
                        removesCount.put(cq, queueCount);
                    }
                    removed += count * cq.averageMessageSize();
                    cq.removeCurrent();
                    queueCount[0] += count;
                    ++queueCount[1];
                }
            }
            for (Map.Entry<ComparableQueue, int[]> entry:
                removesCount.entrySet())
            {
                int[] removes = entry.getValue();
                int removedCount = removes[0];
                if (removedCount > 0L) {
                    logger.info(
                        "Removed: " + removedCount
                        + " docs, " + removes[1]
                        + " terms from LuceneQueue<"
                        + entry.getKey().name() + '>');
                }
            }
            return removesCount;
        }

        private long totalStorageSize() {
            long totalStorageSize = 0L;
            for (LuceneQueue queue : queues.values()) {
                try {
                    long queueSize = queue.storageSize();
                    totalStorageSize += queueSize;
                } catch (IOException ignore) {
                }
            }
            return totalStorageSize;
        }

        @Override
        public void run() {
            long expungeHandicap =
                maxQueueStorageSize - queueStorageSizeDelta / 4;
            // Maps LuceneQueue to last truncate timestamp
            IdentityHashMap<LuceneQueue, Integer> pendingDeletes =
                new IdentityHashMap<>();
            while (true) {
                try {
                    long totalStorageSize = totalStorageSize();
                    logger.info("totalStorageSize: " + totalStorageSize);
                    if (totalStorageSize
                            > maxQueueStorageSize - queueStorageSizeDelta)
                    {
                        logger.info("Running truncate");
                        final ArrayList<ComparableQueue> cQueues =
                            new ArrayList<>(queues.size());
                        try {
                            for (LuceneQueue queue: queues.values()) {
                                YandexComparableQueue cq =
                                    new YandexComparableQueue(
                                        logger,
                                        queue);
                                cQueues.add(cq);
                            }

                            TimePriorityQueue pq =
                                new TimePriorityQueue(cQueues.size());
                            for (ComparableQueue cq : cQueues) {
                                cq.init();
                                Integer oldDeletes = pendingDeletes.get(cq.queue());
                                if (oldDeletes == null) {
                                    pq.add(cq);
                                    logger.info(
                                        "truncate pq init: LuceneQueue<"
                                        + cq.name() + ">, currentTerm: "
                                        + cq.currentTerm());
                                } else {
                                    int deletes = cq.numDeletedDocs();
                                    if (oldDeletes / 2 > deletes) {
                                        pq.add(cq);
                                        logger.info(
                                            "truncate pq init: LuceneQueue<"
                                            + cq.name() + ">, currentTerm: "
                                            + cq.currentTerm()
                                            + ", prev deletes count: "
                                            + oldDeletes
                                            + ", num deleted docs: "
                                            + deletes);
                                    } else {
                                        logger.info(
                                            "Skip truncate for LuceneQueue<"
                                            + cq.name()
                                            + ">, prev deletes count: "
                                            + oldDeletes
                                            + ", num deleted docs: "
                                            + deletes);
                                    }
                                }
                            }
                            long truncateToSize =
                                maxQueueStorageSize - queueStorageSizeDelta * 2;
                            long toRemove = totalStorageSize - truncateToSize;
                            IdentityHashMap<ComparableQueue, int[]> toCommit =
                                truncate(pq, toRemove);
                            for (Map.Entry<ComparableQueue, int[]> entry
                                : toCommit.entrySet())
                            {
                                ComparableQueue cq = entry.getKey();
                                cq.flush();
                                pendingDeletes.put(
                                    cq.queue(),
                                    entry.getValue()[0]);
                            }
                            for (ComparableQueue queue: toCommit.keySet()) {
                                queue.afterTruncate();
                            }
                        } finally {
                            for (ComparableQueue cq : cQueues) {
                                cq.close();
                            }
                        }
                    }
                    while (true) {
                        if (pendingDeletes.isEmpty()) {
                            logger.info("No pending deletes left for expunge");
                            break;
                        }
                        totalStorageSize = totalStorageSize();
                        if (totalStorageSize < expungeHandicap) {
                            break;
                        }
                        logger.info("expungeHandicap: " + expungeHandicap
                            + " has been reached, totalStorageSize: "
                            + totalStorageSize);
                        Iterator<Map.Entry<LuceneQueue, Integer>> iter =
                            pendingDeletes.entrySet().iterator();
                        Map.Entry<LuceneQueue, Integer> entry = iter.next();
                        LuceneQueue bestQueue = entry.getKey();
                        int maxDeletes = entry.getValue().intValue();
                        while (iter.hasNext()) {
                            entry = iter.next();
                            int deletes = entry.getValue().intValue();
                            if (deletes > maxDeletes) {
                                maxDeletes = deletes;
                                bestQueue = entry.getKey();
                            }
                        }
                        logger.info(
                            "Running expunge for LuceneQueue<"
                            + bestQueue.name()
                            + "> with deletes count: " + maxDeletes);
                        pendingDeletes.remove(bestQueue);
                        bestQueue.expunge();
                    }
                    Thread.sleep(30000L);
                } catch (Exception e) {
                    logger.log(Level.WARNING, "Unhandled exception", e);
                }
            }
        }
    }

    private class IndexingThread extends Thread
    {
        public final ArrayBlockingQueue<FutureTask<Void>> queue =
            new ArrayBlockingQueue<FutureTask<Void>>(10);
        public IndexingThread(final int num)
        {
            super("StorageIndexThread-" + num);
            setDaemon(true);
        }

        public void run()
        {
            while( true )
            {
                try {
                    FutureTask<Void> f = queue.take();
                    if (f == null) continue;
                    f.run();
                } catch (Throwable t) {
                    t.printStackTrace();
                }
            }
        }
    }

    private class IndexTask implements Callable<Void>
    {
        final String queuePath;
        final String path;
        final String hash;
        final long ctime;
        final byte[] data;
        final WeighableRunnable callback;
        final long position;

        public IndexTask(
            final long position,
            final String queuePath,
            final String path,
            final String hash,
            final long ctime,
            byte[] data,
            final WeighableRunnable callback)
        {
            this.position = position;
            this.queuePath = queuePath;
            this.path = path;
            this.hash = hash;
            this.ctime = ctime;
            this.data = data;
            this.callback = callback;
        }

        @Override
        public Void call() throws KeeperException {
            addDataInternal(
                position,
                queuePath,
                path,
                hash,
                ctime,
                data,
                callback);
            return null;
        }

    }

    private static final Weigher<CacheKey, byte[]> dataWeigher =
        new Weigher<CacheKey, byte[]>() {
            @Override
            public int weigh(final CacheKey key, final byte[] data) {
                int weight = CACHE_ENTRY_WEIGHT_OVERHEAD;
                weight += AVERAGE_OBJECT_SIZE;
                weight += AVERAGE_OBJECT_SIZE + (key.path().length() << 1);
                if (data != null) {
                    weight += AVERAGE_OBJECT_SIZE + data.length;
                }
                return weight;
            }
        };

    private static final CacheLoader<CacheKey, byte[]> dataLoader =
        new CacheLoader<CacheKey, byte[]>() {
            @Override
            public byte[] load(final CacheKey key)
                throws KeeperException.SystemErrorException,
                    KeeperException.NoNodeException
            {
                return key.load();
            }
        };

    static class CacheKey {
        private final LuceneQueue queue;
        private final String path;
        private AtomicReaderContext[] readers;

        public CacheKey(
            final LuceneQueue queue,
            final String path,
            final AtomicReaderContext[] readers)
        {
            this.queue = queue;
            this.path = path;
            this.readers = readers;
        }

        public String path() {
            return path;
        }

        @Override
        public int hashCode() {
            return path.hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof CacheKey) {
                CacheKey other = (CacheKey) o;
                return other.path.equals(path);
            }
            return false;
        }

        public byte[] load()
            throws KeeperException.SystemErrorException,
                KeeperException.NoNodeException
        {
            try {
                byte[] data = queue.readNodeFromIndex(path, readers);
                if (data == null) {
                    throw new KeeperException.NoNodeException() {
                        @Override
                        public Throwable fillInStackTrace() {
                            return this;
                        }
                    };
                }
                return data;
            } finally {
                readers = null;
            }
        }
    }
}
