package org.apache.zookeeper.server.persistence;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.ref.WeakReference;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.lucene.store.BufferedIndexOutput;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.IndexOutput;

import ru.yandex.parser.string.BooleanParser;
import ru.yandex.util.timesource.TimeSource;

public class WriteLimiter {
    private static final boolean DISABLE_WRITE_LIMITER =
        BooleanParser.INSTANCE.apply(
            System.getProperty(
                "zoolooser.disable-write-limiter",
                "false"));

    //TODO: Remove singleton
    public static final WriteLimiter INSTANCE =
        new WriteLimiter(Integer.MAX_VALUE);

    public static final int DEFAULT_PRIO = 100;
    public static final int FLUSH_PRIO = 90;
    public static final int SNAPSHOT_PRIO = 80;
    public static final int MERGE_PRIO = 1;

    private static final int MB_BYTES = 1024 * 1024;
    private static final int SEC = 1000;
    private static final int LIMITER_RESOLUTION = 100;
    private static final int RATE_DIVISOR = SEC / LIMITER_RESOLUTION;


    private final PriorityBlockingQueue<Writer> prioQueue =
        new PriorityBlockingQueue<>();
    private long bytesLimitPerTick;
    private final Object lock = new Object();
    private final AtomicLong bytesCountDown = new AtomicLong(0);
    private final AtomicLong bytesCopied = new AtomicLong(0);
    private long prevRateCheck = 0;
    private long lastCountDownResetTime = 0;

    private WriteLimiter(final int rateLimitMb) {
        this.bytesLimitPerTick = calcPerTick(rateLimitMb);;
        bytesCountDown.set(bytesLimitPerTick);
        TimerHelper timer = new TimerHelper(this);
        timer.setDaemon(true);
        timer.start();
    }

    private long calcPerTick(final int limitMb) {
        return ((long) limitMb * (long) MB_BYTES) / (long) RATE_DIVISOR;
    }

    public void rateLimitMb(final int rateLimitMb) {
        this.bytesLimitPerTick = calcPerTick(rateLimitMb);;
        bytesCountDown.set(bytesLimitPerTick);
        System.err.println("BytesLimitPerTick: " + bytesLimitPerTick);
    }

    public static OutputStream wrapOutputStream(
        final OutputStream stream)
    {
        if (DISABLE_WRITE_LIMITER) {
            return stream;
        }
        return new RateLimitOutputStream(
            stream,
            INSTANCE,
            new Writer(SNAPSHOT_PRIO));
    }

    public static IndexOutput wrapIndexOutput(
        final BufferedIndexOutput out)
//        final IOContext context)
    {
        if (DISABLE_WRITE_LIMITER) {
            return out;
        }
        final int prio;
//        if (context.context == IOContext.Context.FLUSH) {
//            prio = FLUSH_PRIO;
//        } else if (context.context == IOContext.Context.DEFAULT) {
//            prio = DEFAULT_PRIO;
//        } else {
            prio = MERGE_PRIO;
//        }
        return new RateLimitIndexOutput(out, INSTANCE, new Writer(prio));
    }

    public double currentRate() {
        final long currentTime = TimeSource.INSTANCE.currentTimeMillis();
        final double bytesCopied = this.bytesCopied.getAndSet(0);
        final double elapsedTime =
            Math.max(
                0.000001,
                (double) (currentTime - prevRateCheck) / (double)SEC);
        prevRateCheck = currentTime;
        final double rate = bytesCopied / elapsedTime;
        return rate / MB_BYTES;
    }

    protected void rateLimit(final Writer writer, final int len) {
        bytesCopied.addAndGet(len);
        final long bytesAllowed = bytesCountDown.addAndGet(-len);
        if (bytesAllowed < 0) {
            synchronized (writer) {
                prioQueue.offer(writer);
                try {
                    writer.wait();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    public void timerTick() {
        if (bytesCountDown.get() < 0) {
            if (bytesCountDown.addAndGet(bytesLimitPerTick) > 0) {
                notifyWaiters();
            }
        } else {
            bytesCountDown.set(bytesLimitPerTick);
            notifyWaiters();
        }
    }

    private void notifyWaiters() {
        Writer toWake;
        while ((toWake = prioQueue.poll()) != null) {
            synchronized (toWake) {
                toWake.notify();
            }
        }
    }

    private static class RateLimitOutputStream extends FilterOutputStream {
        private final WriteLimiter limiter;
        private final Writer writer;

        public RateLimitOutputStream(
            final OutputStream out,
            final WriteLimiter limiter,
            final Writer writer)
        {
            super(out);
            this.limiter = limiter;
            this.writer = writer;
        }

        @Override
        public void write(final byte[] b) throws IOException {
            limiter.rateLimit(writer, b.length);
            out.write(b);
        }

        @Override
        public void write(final byte[] b, final int off, final int len)
            throws IOException
        {
            limiter.rateLimit(writer, len);
            out.write(b, off, len);
        }

        @Override
        public void write(final int b) throws IOException {
            limiter.rateLimit(writer, 1);
            out.write(b);
        }
    }

    private static class RateLimitIndexOutput extends BufferedIndexOutput {
        private final BufferedIndexOutput out;
        private final WriteLimiter limiter;
        private final Writer writer;

        public RateLimitIndexOutput(
            final BufferedIndexOutput out,
            final WriteLimiter limiter,
            final Writer writer)
        {
            this.out = out;
            this.limiter = limiter;
            this.writer = writer;
        }

        @Override
        public void flushBuffer(
            final byte[] b,
            final int offset,
            final int size)
            throws IOException
        {
            limiter.rateLimit(writer, size);
            out.flushBuffer(b, offset, size);
        }

        @Override
        public void seek(final long pos) throws IOException {
            super.seek(pos);
            out.seek(pos);
        }

        @Override
        public long length() throws IOException {
            return out.length();
        }

        @Override
        public void setLength(long length) throws IOException {
            out.setLength(length);
        }

        @Override
        public void close() throws IOException {
            super.close();
            out.close();
        }
    }

    private static class TimerHelper extends Thread {
        private final WeakReference<WriteLimiter> limiterRef;
        public TimerHelper(final WriteLimiter limiter) {
            super("TimerHelper");
            this.limiterRef = new WeakReference<>(limiter);
        }

        @Override
        public void run() {
            while (true) {
                WriteLimiter limiter = limiterRef.get();
                if (limiter == null) {
                    return;
                }
                try {
                    Thread.sleep(LIMITER_RESOLUTION);
                    limiter.timerTick();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private static class Writer implements Comparable<Writer> {
        private final int prio;

        public Writer(final int prio) {
            //normalize
            this.prio = prio;
        }

        @Override
        public int compareTo(final Writer other) {
            return Integer.compare(other.prio, prio);
        }
    }
}
