#include <boost/asio.hpp>
#include <boost/variant.hpp>
#include <iostream>
#include <boost/format.hpp>
#include <yadns/ares_resolver_service.h>
#include <yadns/basic_dns_resolver.h>

typedef
boost::variant<
    basic_dns_resolver<ares_resolver_service>::iterator_a
  , basic_dns_resolver<ares_resolver_service>::iterator_aaaa
  , basic_dns_resolver<ares_resolver_service>::iterator_ptr
  , boost::asio::ip::tcp::resolver::iterator
> basic_resolver_iterator;

std::string rev_order_av4_str(const boost::asio::ip::address_v4& a)
{
  boost::asio::ip::address_v4::bytes_type bytes = a.to_bytes();
  return str(boost::format("%1%.%2%.%3%.%4%.in-addr.arpa.")
      % static_cast<int>(bytes[3])
      % static_cast<int>(bytes[2])
      % static_cast<int>(bytes[1])
      % static_cast<int>(bytes[0])
    );
}

std::string rev_order_av6_str(const boost::asio::ip::address_v6& a)
{
  std::ostringstream os;
  boost::asio::ip::address_v6::bytes_type bytes = a.to_bytes();
  const char a16[] = "0123456789abcdef";
  BOOST_REVERSE_FOREACH(int i, bytes)
    os << a16[i%16] << "." << a16[i/16] << ".";
  os << "ip6.arpa.";
  return os.str();
}

struct resolver_iterator
{
  typedef boost::asio::ip::tcp protocol_type;
  typedef boost::asio::ip::basic_resolver_entry<protocol_type> value_type;
  typedef boost::asio::ip::basic_resolver_query<protocol_type> query;
  typedef protocol_type::endpoint endpoint_type;

  basic_resolver_iterator v_;
  std::string host_;
  std::string service_;
  endpoint_type e_;

  struct dereference_op : boost::static_visitor<value_type>
  {
    const resolver_iterator* that_;

    dereference_op(const resolver_iterator* it)
      : that_(it)
    {}

    typedef basic_dns_resolver<ares_resolver_service> resolver;

    value_type operator()(const resolver::iterator_a& it) const
    {
      boost::asio::ip::address addr =
        boost::asio::ip::address::from_string(*it);
      return value_type(
        endpoint_type(boost::asio::ip::address::from_string(*it),
            that_->e_.port())
        , that_->host_, that_->service_);
    }

    value_type operator()(const resolver::iterator_aaaa& it) const
    {
      boost::asio::ip::address addr =
        boost::asio::ip::address::from_string(*it);
      return value_type(
        endpoint_type(boost::asio::ip::address::from_string(*it),
            that_->e_.port())
        , that_->host_, that_->service_);
    }

    value_type operator()(const resolver::iterator_ptr& it) const
    {
      return value_type(that_->e_, *it, that_->service_);
    }

    value_type
    operator()(const boost::asio::ip::tcp::resolver::iterator& it) const
    {
      return *it;
    }
  };

  struct isend_op : boost::static_visitor<bool>
  {
    typedef basic_dns_resolver<ares_resolver_service> resolver;

    bool operator()(const resolver::iterator_a& it) const
    {
      return it == resolver::iterator_a();
    }

    bool operator()(const resolver::iterator_aaaa& it) const
    {
      return it == resolver::iterator_aaaa();
    }

    bool operator()(const resolver::iterator_ptr& it) const
    {
      return it == resolver::iterator_ptr();
    }

    bool operator()(boost::asio::ip::tcp::resolver::iterator it) const
    {
      return it == boost::asio::ip::tcp::resolver::iterator();
    }
  };

  struct notequal_op : boost::static_visitor<bool>
  {
    const resolver_iterator* that_;

    notequal_op(const resolver_iterator* it)
      : that_(it)
    {}

    typedef basic_dns_resolver<ares_resolver_service> resolver;

    template <typename Iterator>
    bool operator()(const Iterator& rh) const
    {
      try
      {
        isend_op op;
        bool lh_is_end = boost::apply_visitor(op, that_->v_);
        bool rh_is_end = op(rh);

        if (lh_is_end || rh_is_end)
          return lh_is_end != rh_is_end;

        const auto& lh =
          boost::get<Iterator>(that_->v_);
        return lh != rh;
      } catch (...) { return false; }
    }
  };

  struct incr_op : boost::static_visitor<void>
  {
    typedef basic_dns_resolver<ares_resolver_service> resolver;

    template <typename Iterator>
    void operator()(Iterator& rh) const { ++rh; }
  };

  bool operator != (const resolver_iterator& lh) const
  {
    notequal_op op(this);
    return boost::apply_visitor(op, lh.v_);
  }

  resolver_iterator operator++(int)
  {
    resolver_iterator saved = *this;
    incr_op op;
    boost::apply_visitor(op, v_);
    return saved;
  }

  value_type operator*() const
  {
    dereference_op op(this);
    return boost::apply_visitor(op, v_);
  }
};

class resolver_adapter
{
  typedef basic_dns_resolver<ares_resolver_service> resolver;

  resolver r_;

public:
  typedef resolver_iterator iterator;
  typedef boost::asio::ip::tcp protocol_type;
  typedef protocol_type::endpoint endpoint_type;
  typedef boost::asio::ip::basic_resolver_query<protocol_type> query;
  typedef void service_type;

  resolver_adapter(boost::asio::io_service& ios)
    : r_(ios)
  {}

  template <typename Handler>
  void async_resolve(query q, Handler h)
  {
    int port = -1;
    try { port = std::stoi(q.service_name()); }
    catch (...) {}

    // Fallback to asio if service name is not numeric or hostname is empty
    if (port < 0 || q.host_name().empty())
    {
      typedef boost::asio::ip::tcp::resolver asio_resolver_t;
      boost::asio::io_service& ios = r_.get_io_service();
      std::shared_ptr<asio_resolver_t> r(new asio_resolver_t(ios));
      return r->async_resolve(q,
          [h, q, r](const boost::system::error_code& ec,
            typename asio_resolver_t::iterator it)
          {
            basic_resolver_iterator v(it);
            iterator it_adapted = { v, q.host_name(), q.service_name() };
            return h(ec, it_adapted);
          }
        );
    }

    if (q.hints().ai_family == BOOST_ASIO_OS_DEF(AF_INET6))
    {
      std::function<void(const boost::system::error_code& ec,
          typename resolver::iterator_aaaa it)> handler =
        [h, q, port](const boost::system::error_code& ec,
            typename resolver::iterator_aaaa it)
        {
          basic_resolver_iterator v(it);
          endpoint_type ep(boost::asio::ip::tcp::v6(), port);
          iterator it_adapted = { v, q.host_name(), q.service_name(), ep };
          return h(ec, it_adapted);
        };
      return r_.async_resolve_aaaa(q.host_name(), handler);
    }
    else
    {
      std::function<void(const boost::system::error_code& ec,
          typename resolver::iterator_a it)> handler =
        [h, q, port](const boost::system::error_code& ec,
            typename resolver::iterator_a it)
        {
          basic_resolver_iterator v(it);
          endpoint_type ep(boost::asio::ip::tcp::v4(), port);
          iterator it_adapted = { v, q.host_name(), q.service_name(), ep };
          return h(ec, it_adapted);
        };
      return r_.async_resolve_a(q.host_name(), handler);
    }
  }

  template <typename Handler>
  void async_resolve(endpoint_type e, Handler h)
  {
    boost::asio::ip::address addr (e.address());
    // Fallback to asio if address is unspecified
    if (addr.is_unspecified())
    {
      typedef boost::asio::ip::tcp::resolver asio_resolver_t;
      boost::asio::io_service& ios = r_.get_io_service();
      std::shared_ptr<asio_resolver_t> r(new asio_resolver_t(ios));
      return r->async_resolve(e,
          [h, r](const boost::system::error_code& ec,
            typename asio_resolver_t::iterator it)
          {
            basic_resolver_iterator v(it);
            iterator it_adapted = { v };
            return h(ec, it_adapted);
          }
        );
    }
    else
    {
      std::function<void(const boost::system::error_code& ec,
          typename resolver::iterator_ptr it)> handler =
        [h, e](const boost::system::error_code& ec,
            typename resolver::iterator_ptr it)
        {
          basic_resolver_iterator v(it);
          iterator it_adapted = { v, "", "", e };
          return h(ec, it_adapted);
        };

      if (addr.is_v4())
        return r_.async_resolve_ptr(rev_order_av4_str(addr.to_v4()), handler);
      else
        return r_.async_resolve_ptr(rev_order_av6_str(addr.to_v6()), handler);
    }
  }
};

template <typename Resolver>
void test_resolver(Resolver& r, std::ostream& os)
{
  typedef typename Resolver::endpoint_type endpoint_t;
  typedef typename Resolver::iterator iterator_t;
  typedef typename Resolver::protocol_type protocol_t;
  typedef typename Resolver::query query_t;
  typedef typename Resolver::service_type service_t;
  typedef typename iterator_t::value_type resolver_entry_t;
  typedef boost::asio::ip::address address_t;

  struct test_resolver_helper
  {
    std::ostream& os_;
    std::string descr_;

    void operator()(const boost::system::error_code& ec, iterator_t it) const
    {
      os_ << descr_ << ": ec: " << ec.message() << std::endl;
      if (ec)
        return;

      for (int i=0; it != iterator_t(); ++i)
      {
        resolver_entry_t entry = *it++;
        os_ << descr_ << ": " << i << ": endpoint: " << entry.endpoint()
            << std::endl
            << descr_ << ": " << i << ": host_name: " << entry.host_name()
            << std::endl
            << descr_ << ": " << i << ": service_name: " << entry.service_name()
            << std::endl;
      }
    }

    test_resolver_helper& operator()(const char* descr)
    {
      descr_ = descr;
      return* this;
    }

  } handle_resolve({os});

  // Part 1: Forward resolution tests
  r.async_resolve(query_t("http"),
      handle_resolve("async_resolve (1.1) test"));

  r.async_resolve(
    query_t("gmail.com", "http")
    , handle_resolve("async_resolve (1.2) test"));

  r.async_resolve(
    query_t("gmail.com", "80")
    , handle_resolve("async_resolve (1.3) test"));

  r.async_resolve(
    query_t(protocol_t::v4(), "gmail.com", "80",
        query_t::address_configured | query_t::numeric_service)
    , handle_resolve("async_resolve (1.4) test"));

  r.async_resolve(
    query_t(protocol_t::v6(), "gmail.com", "80")
    , handle_resolve("async_resolve (1.5) test"));

  r.async_resolve(
    query_t("hampers", "ssh")
    ,  handle_resolve("async_resolve (1.6) test (/etc/hosts)"));

  // Part2: reverse resolution tests
  r.async_resolve(
    endpoint_t(protocol_t::v4(), 25)
    , handle_resolve("async_resolve (2.1) test"));

  r.async_resolve(
    endpoint_t(protocol_t::v6(), 25)
    , handle_resolve("async_resolve (2.2) test"));

  r.async_resolve(
    endpoint_t(address_t::from_string("74.125.237.213"), 25)
    , handle_resolve("async_resolve (2.3) test"));

  r.async_resolve(
    endpoint_t(address_t::from_string("2404:6800:4006:804::1015"), 25)
    , handle_resolve("async_resolve (2.4) test"));
}

int main()
{
  boost::asio::io_service ios;
  std::ostringstream o1, o2;

  boost::asio::ip::tcp::resolver r1(ios);
  test_resolver(r1, o1);

  resolver_adapter r2(ios);
  test_resolver(r2, o2);

  ios.run();

  std::cout  << o1.str() << "--\n"
             << o2.str() << std::endl;

  return 0;
}
