#include "log.h"
#include <boost/regex.hpp>
#include <boost/algorithm/string/predicate.hpp>

using namespace lg;

struct logger::implementation
{
};

logger::logger()
{}

logger::~logger()
{}

logger* logger::instance()
{
    static logger l;
    return &l;
}

namespace
{
#define HANDLE_FACILITY(F)                      \
    if (boost::istarts_with(facility, #F))      \
        return sinks::syslog::F

    sinks::syslog::facility parse_syslog_facility(const std::string& facility)
    {
        HANDLE_FACILITY(local0);
        HANDLE_FACILITY(local1);
        HANDLE_FACILITY(local2);
        HANDLE_FACILITY(local3);
        HANDLE_FACILITY(local4);
        HANDLE_FACILITY(local5);
        HANDLE_FACILITY(local6);
        HANDLE_FACILITY(local7);
        HANDLE_FACILITY(kernel);
        HANDLE_FACILITY(user);
        HANDLE_FACILITY(mail);
        HANDLE_FACILITY(daemon);
        HANDLE_FACILITY(security0);
        HANDLE_FACILITY(syslogd);
        HANDLE_FACILITY(printer);
        HANDLE_FACILITY(news);
        HANDLE_FACILITY(uucp);
        HANDLE_FACILITY(clock0);
        HANDLE_FACILITY(security1);
        HANDLE_FACILITY(ftp);
        HANDLE_FACILITY(ntp);
        HANDLE_FACILITY(log_audit);
        HANDLE_FACILITY(log_alert);
        HANDLE_FACILITY(clock1);

        throw std::runtime_error(
            std::string("bad syslog facility").append(facility));
    }
}

void logger::add_sink(const std::string& url)
{
    typedef std::string::const_iterator iterator;
    boost::match_results<iterator> m;
    iterator beg = url.begin();
    iterator end = url.end();

    boost::regex expr("^([[:alpha:]]+)(://([[:alnum:]/\\._]*))?",
            boost::regex::icase);
    if (boost::regex_match(beg, end, m, expr, boost::match_default))
    {
        const std::string& proto = m[1];
        const std::string& param = m[3];
        if (boost::istarts_with(proto, "syslog"))
        {
            if (!param.empty())
                return add_syslog_sink(parse_syslog_facility(param));
        }
        else if (boost::istarts_with(proto, "file"))
        {
            if (!param.empty())
                return add_file_sink(param);
        }
        else if (boost::istarts_with(proto, "stdout"))
        {
            return add_ostream_sink(std::cout);
        }
        else if (boost::istarts_with(proto, "stderr"))
        {
            return add_ostream_sink(std::cerr);
        }
        else if (boost::istarts_with(proto, "stdlog"))
        {
            return add_ostream_sink(std::clog);
        }
    }

    throw std::runtime_error(std::string("bad sink: ").append(url));
}

void logger::add_file_sink(const std::string& filename)
{
    typedef sinks::text_file_backend backend_t;
    typedef sinks::asynchronous_sink< sinks::text_file_backend > sink_t;
    std::ios_base::openmode open_mode = std::ios_base::app | std::ios_base::out;

    boost::shared_ptr< logging::core > core = logging::core::get();
    boost::shared_ptr<  backend_t > backend =
        boost::make_shared<  backend_t > (
            logging::keywords::file_name = filename,
            logging::keywords::open_mode = open_mode);
    backend->set_formatter(fmt::stream
            << fmt::date_time("TimeStamp", keywords::format = "%b %d %H:%M:%S")
            << " " << fmt::message());
    boost::shared_ptr< sink_t > sink = boost::make_shared<sink_t>(backend);
    core->add_sink(sink);
    backend->auto_flush(true);

    logging::core::get()->add_global_attribute(
        "TimeStamp",
        attrs::local_clock());
}

void logger::add_syslog_sink(sinks::syslog::facility facility)
{
    typedef sinks::syslog_backend backend_t;
    typedef sinks::asynchronous_sink< backend_t > sink_t;

    boost::shared_ptr<logging::core> core = logging::core::get();
    boost::shared_ptr<backend_t> backend =
        boost::make_shared<backend_t> (
            keywords::facility = facility,
            keywords::use_impl = sinks::syslog::native);

    backend->set_severity_mapper(
        sinks::syslog::direct_severity_mapping<int>("Severity"));
    core->add_sink(boost::make_shared< sink_t >(backend));
}

void logger::add_ostream_sink(std::ostream& os)
{
    typedef sinks::text_ostream_backend backend_t;
    typedef sinks::asynchronous_sink< sinks::text_ostream_backend > sink_t;

    boost::shared_ptr< logging::core > core = logging::core::get();
    boost::shared_ptr<  backend_t > backend =
        boost::make_shared<  backend_t > ();
    backend->add_stream(
        boost::shared_ptr< std::ostream >(&os, logging::empty_deleter()));

    backend->set_formatter(fmt::stream << fmt::date_time("TimeStamp",
                    keywords::format = "%H:%M:%S")
            << " " << fmt::message());
    boost::shared_ptr< sink_t > sink = boost::make_shared<sink_t>(backend);
    core->add_sink(sink);
    backend->auto_flush(true);

    logging::core::get()->add_global_attribute(
        "TimeStamp",
        attrs::local_clock());
}
