#ifndef YADNS_LOG_H
#define YADNS_LOG_H

#include <boost/scoped_ptr.hpp>
#include <boost/log/sinks.hpp>
#include <boost/log/attributes.hpp>
#include <boost/log/common.hpp>
#include <boost/log/core.hpp>
#include <boost/log/filters.hpp>
#include <boost/log/formatters.hpp>
#include <boost/log/sources/logger.hpp>
#include <boost/parameter/keyword.hpp>
#include <boost/thread.hpp>
#include <string>
#include <ostream>
#include <boost/filesystem/fstream.hpp>
#include <boost/noncopyable.hpp>

namespace lg
{

    namespace logging = boost::log;
    namespace fmt = boost::log::formatters;
    namespace flt = boost::log::filters;
    namespace sinks = boost::log::sinks;
    namespace attrs = boost::log::attributes;
    namespace src = boost::log::sources;
    namespace keywords = boost::log::keywords;

    const sinks::syslog::level error = sinks::syslog::error;
    const sinks::syslog::level info = sinks::syslog::info;
    const sinks::syslog::level debug = sinks::syslog::debug;

} // namespace lg

class logger :
    private boost::noncopyable
{
    logger();
    ~logger();

public:
    /**
     * Global logger accessor method.
     * Note: the first invocation should be made from within a single thread.
     */
    static logger* instance();

    /**
     * Add a sink to log to.
     * This can be one of the following:
     * 1. file://filename
     * 2. syslog://facility
     * 3. stdout
     * 4. stderr
     * 5. stdlog
     */
    void add_sink(const std::string& sink_url = "stdlog");

    void add_file_sink(const std::string& filename);

    void add_syslog_sink(lg::sinks::syslog::facility facility);

    void add_ostream_sink(std::ostream& os);

    // This is intended to be accessed through L_* macros.
    lg::src::logger_mt l_;

private:
    class implementation;
    boost::scoped_ptr<implementation> impl_;
};

// Macros to log error, information and error messages.
# ifndef YADNS_DISABLE_LOGGING

#define L_ERR(x)     { BOOST_LOG_SEV(logger::instance()->l_, lg::error) << x; }
#define L_INFO(x)    { BOOST_LOG_SEV(logger::instance()->l_, lg::info) << x; }
#define L_DEBUG(x)   { BOOST_LOG_SEV(logger::instance()->l_, lg::debug) << x; }

# else

#define L_ERR(x)
#define L_INFO(x)
#define L_DEBUG(x)

#endif

#endif /* YADNS_LOG_H */
