#include <iostream>
#include <fstream>

#include <boost/date_time.hpp>
#include <boost/program_options.hpp>
#include <boost/thread.hpp>

#include <yadns/ares_resolver_service.h>
#include <yadns/basic_dns_resolver.h>

#include "log.h"

typedef basic_dns_resolver<ares_resolver_service> resolver;

template <class Cont>
void handle_resolve_a(const boost::system::error_code& ec,
    resolver::iterator_a it, const std::string& host, Cont cont)
{
  L_INFO("*** handle_resolve_a: " << host << ": " << ec.message());
  while (it != resolver::iterator_a())
    L_INFO(host << " A: " << *it++);
  return cont();
}

template <class Cont>
void handle_resolve_aaaa(const boost::system::error_code& ec,
    resolver::iterator_aaaa it, const std::string& host, Cont cont)
{
  L_INFO("*** handle_resolve_aaaa: " << host << ": " << ec.message());
  while (it != resolver::iterator_aaaa())
    L_INFO(host << " AAAA: " << *it++);
  return cont();
}

template <class Cont>
void handle_resolve_ptr(const boost::system::error_code& ec,
    resolver::iterator_ptr it, const std::string& host, Cont cont)
{
  L_INFO("*** handle_resolve_ptr: " << host << ": " << ec.message());
  while (it != resolver::iterator_ptr())
    L_INFO(host << " PTR: " << *it++);
  return cont();
}

template <class Cont>
void handle_resolve_mx(const boost::system::error_code& ec,
    resolver::iterator_mx it, const std::string& host, Cont cont)
{
  L_INFO("*** handle_resolve_mx: " << host << ": " << ec.message());
  while (it != resolver::iterator_mx())
    L_INFO(host << " MX: " << *it++);
  return cont();
}

template <class Cont>
void handle_resolve_txt(const boost::system::error_code& ec,
    resolver::iterator_txt it, const std::string& host, Cont cont)
{
  L_INFO("*** handle_resolve_txt: " << host << ": " << ec.message());
  while (it != resolver::iterator_txt())
    L_INFO(host << " TXT: " << *it++);
  return cont();
}

template <class Cont>
void async_resolve_a(resolver& r, const std::string& host, Cont cont)
{
  r.async_resolve_a(host,
      boost::bind(handle_resolve_a<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_aaaa(resolver& r, const std::string& host, Cont cont)
{
  r.async_resolve_aaaa(host,
      boost::bind(handle_resolve_aaaa<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_ptr(resolver& r, const std::string& host, Cont cont)
{
  r.async_resolve_ptr(host,
      boost::bind(handle_resolve_ptr<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_mx(resolver& r, const std::string& host, Cont cont)
{
  r.async_resolve_mx(host,
      boost::bind(handle_resolve_mx<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_txt(resolver& r, const std::string& host, Cont cont)
{
  r.async_resolve_txt(host,
      boost::bind(handle_resolve_txt<Cont>, _1, _2, host, cont));
}

// -----------
struct resolv_parameters
{
  resolv_parameters()
    : tcnt(2)
    , rcnt(1)
    , max_queries_pending(200)
  {}
  int tcnt;
  int rcnt;
  int max_queries_pending;
};

struct foo
  : public boost::enable_shared_from_this<foo>
  , private boost::noncopyable
{
  typedef boost::shared_ptr<resolver> resolver_ptr_t;

  resolv_parameters opt_;
  boost::asio::io_service ios_;
  boost::asio::io_service::strand strand_;
  std::vector<resolver_ptr_t> resolvers_;
  std::vector<resolver_ptr_t>::iterator resolver_it_;
  boost::thread_group thr_;
  int queries_pending_;
  int total_queries_done_;
  std::istream& in_;

  enum { n_of_query_types = 5 };

  foo(const resolv_parameters& opt, std::istream& in)
    : opt_(opt)
    , ios_()
    , strand_(ios_)
    , resolvers_()
    , resolver_it_()
    , thr_()
    , queries_pending_(0)
    , total_queries_done_(0)
    , in_(in)
  {
    assert(opt_.rcnt > 0);

    for (int i=0; i<opt_.rcnt; ++i)
      resolvers_.push_back(resolver_ptr_t(
            new resolver(ios_)));

    resolver_it_ = resolvers_.begin();
  }

  int run()
  {
    assert(opt_.tcnt > 0);

    strand_.dispatch(boost::bind(&foo::step, this));

    for (int i=0; i< opt_.tcnt; ++i)
      thr_.create_thread(boost::bind(&boost::asio::io_service::run,
              &ios_));
    thr_.join_all();
    return total_queries_done_;
  }

  struct cont
  {
    boost::shared_ptr<foo> f_;

    explicit cont(boost::shared_ptr<foo> f)
      : f_(f)
    {}

    void operator()()
    {
      f_->handle_query_done();
    }
  };

  inline cont make_cont() { return cont(shared_from_this()); }

  void handle_query_done()
  {
    total_queries_done_++;
    if (--queries_pending_ + n_of_query_types <=
        std::max(static_cast<int>(n_of_query_types),
            opt_.max_queries_pending))
      step();
  }

  void handle_close_socket(boost::shared_ptr<boost::asio::ip::udp::socket> ptr)
  {
  }

  void step()
  {
    std::string host;
    while (getline(in_, host))
    {
      if (resolver_it_ == resolvers_.end())
        resolver_it_ = resolvers_.begin();

      if (!host.empty())
      {
        resolver& r = **resolver_it_++;
        async_resolve_a(r, host, strand_.wrap(make_cont()));
        async_resolve_aaaa(r, host, strand_.wrap(make_cont()));
        async_resolve_ptr(r, host, strand_.wrap(make_cont()));
        async_resolve_mx(r, host, strand_.wrap(make_cont()));
        async_resolve_txt(r, host, strand_.wrap(make_cont()));
      }

      queries_pending_ += n_of_query_types;

      if (queries_pending_ + n_of_query_types >
          std::max(static_cast<int>(n_of_query_types),
              opt_.max_queries_pending))
        return;
    }
  }
};

int main(int argc, char** argv)
{
  resolv_parameters p;
  std::string log;

  namespace bpo =  boost::program_options;
  boost::program_options::options_description cmd_opt("cmd line options");
  cmd_opt.add_options()
    ("help,h", "produce help message")
    ("threads,r", bpo::value<int>(&p.tcnt)->default_value(4),
        "thread count")
    ("resolvers,s", bpo::value<int>(&p.rcnt)->default_value(4),
        "resolver count")
    ("max_queries,q", bpo::value<int>(
      &p.max_queries_pending)->default_value(500),
        "max outstanding queries")
    ("log,l", bpo::value<std::string>(&log)->default_value(
      "file:///tmp/list"),
        "specifies where to log; possible options are: "
        "file://filename, syslog://facility, stdout, stderr, stdlog")
    ;
  boost::program_options::variables_map vm;
  try
  {
    boost::program_options::store(
      boost::program_options::command_line_parser(
        argc, argv).options(cmd_opt).run(), vm);
    boost::program_options::notify(vm);
    if (vm.count("help"))
    {
      std::cout << cmd_opt << std::endl;
      return 1;
    }
  }
  catch(const std::exception& e)
  {
    std::cerr << "bad options: " << e.what() << std::endl;
    return -1;
  }

  try
  {
    // Setup logging.
    logger::instance()->add_sink(log);
  }
  catch (std::exception& e)
  {
    std::cerr << "error initializing log: " << e.what();
    return -1;
  }


  boost::posix_time::ptime tm =
    boost::posix_time::microsec_clock::local_time();

  int tq = boost::shared_ptr<foo>(new foo(p, std::cin))->run();

  std::cout << tq << " queries; time elapsed: "
            << boost::posix_time::microsec_clock::local_time()-tm << std::endl;

  L_INFO("bye");

  return 0;
}
