#ifndef _ADNS_RESOLVER_ADAPTER_H_
#define _ADNS_RESOLVER_ADAPTER_H_

#include <boost/asio.hpp>
#include <boost/variant.hpp>
#include <boost/bind/protect.hpp>
#include <string>
#include <iostream>
#include <memory>
#include <boost/format.hpp>
#include <yadns/ares_resolver_service.h>
#include <yadns/basic_dns_resolver.h>

typedef
boost::variant<
    basic_dns_resolver<ares_resolver_service>::iterator_a
    , basic_dns_resolver<ares_resolver_service>::iterator_aaaa
    , basic_dns_resolver<ares_resolver_service>::iterator_ptr
    , boost::asio::ip::tcp::resolver::iterator
    > basic_resolver_iterator;

inline std::string rev_order_av4_str(const boost::asio::ip::address_v4& a)
{
  boost::asio::ip::address_v4::bytes_type bytes = a.to_bytes();
  return str(boost::format("%1%.%2%.%3%.%4%.in-addr.arpa.")
      % static_cast<int>(bytes[3])
      % static_cast<int>(bytes[2])
      % static_cast<int>(bytes[1])
      % static_cast<int>(bytes[0])
    );
}

inline std::string rev_order_av6_str(const boost::asio::ip::address_v6& a)
{
  std::ostringstream os;
  boost::asio::ip::address_v6::bytes_type bytes = a.to_bytes();
  const char a16[] = "0123456789abcdef";
  BOOST_REVERSE_FOREACH(int i, bytes)
    os << a16[i%16] << "." << a16[i/16] << ".";
  os << "ip6.arpa.";
  return os.str();
}

struct resolver_iterator
{
  typedef boost::asio::ip::tcp protocol_type;
  typedef boost::asio::ip::basic_resolver_entry<protocol_type> value_type;
  typedef boost::asio::ip::basic_resolver_query<protocol_type> query;
  typedef protocol_type::endpoint endpoint_type;

  basic_resolver_iterator v_;
  std::string host_;
  std::string service_;
  endpoint_type e_;

  struct dereference_op : boost::static_visitor<value_type>
  {
    const resolver_iterator* that_;

    dereference_op(const resolver_iterator* it)
      : that_(it)
    {}

    typedef basic_dns_resolver<ares_resolver_service> resolver;

    value_type operator()(const resolver::iterator_a& it) const
    {
      boost::asio::ip::address addr =
        boost::asio::ip::address::from_string(*it);
      return value_type(
        endpoint_type(boost::asio::ip::address::from_string(*it),
            that_->e_.port())
        , that_->host_, that_->service_);
    }

    value_type operator()(const resolver::iterator_aaaa& it) const
    {
      boost::asio::ip::address addr =
        boost::asio::ip::address::from_string(*it);
      return value_type(
        endpoint_type(boost::asio::ip::address::from_string(*it),
            that_->e_.port())
        , that_->host_, that_->service_);
    }

    value_type operator()(const resolver::iterator_ptr& it) const
    {
      return value_type(that_->e_, *it, that_->service_);
    }

    value_type
    operator()(const boost::asio::ip::tcp::resolver::iterator& it) const
    {
      return *it;
    }
  };

  struct isend_op : boost::static_visitor<bool>
  {
    typedef basic_dns_resolver<ares_resolver_service> resolver;

    bool operator()(const resolver::iterator_a& it) const
    {
      return it == resolver::iterator_a();
    }

    bool operator()(const resolver::iterator_aaaa& it) const
    {
      return it == resolver::iterator_aaaa();
    }

    bool operator()(const resolver::iterator_ptr& it) const
    {
      return it == resolver::iterator_ptr();
    }

    bool operator()(boost::asio::ip::tcp::resolver::iterator it) const
    {
      return it == boost::asio::ip::tcp::resolver::iterator();
    }
  };

  struct notequal_op : boost::static_visitor<bool>
  {
    const resolver_iterator* that_;

    notequal_op(const resolver_iterator* it)
      : that_(it)
    {}

    typedef basic_dns_resolver<ares_resolver_service> resolver;

    template <typename Iterator>
    bool operator()(const Iterator& rh) const
    {
      try
      {
        isend_op op;
        bool lh_is_end = boost::apply_visitor(op, that_->v_);
        bool rh_is_end = op(rh);

        if (lh_is_end || rh_is_end)
          return lh_is_end != rh_is_end;

        const auto& lh =
          boost::get<Iterator>(that_->v_);
        return lh != rh;
      } catch (...) { return false; }
    }
  };

  struct incr_op : boost::static_visitor<void>
  {
    typedef basic_dns_resolver<ares_resolver_service> resolver;

    template <typename Iterator>
    void operator()(Iterator& rh) const { ++rh; }
  };

  bool operator != (const resolver_iterator& lh) const
  {
    notequal_op op(this);
    return boost::apply_visitor(op, lh.v_);
  }

  resolver_iterator operator++(int)
  {
    resolver_iterator saved = *this;
    incr_op op;
    boost::apply_visitor(op, v_);
    return saved;
  }

  value_type operator*() const
  {
    dereference_op op(this);
    return boost::apply_visitor(op, v_);
  }
};

struct resolver_adapter_traits
{
  typedef resolver_iterator iterator;
  typedef boost::asio::ip::tcp protocol_type;
  typedef protocol_type::endpoint endpoint_type;
  typedef boost::asio::ip::basic_resolver_query<protocol_type> query;
  typedef void service_type;

  typedef basic_dns_resolver<ares_resolver_service> resolver;
  typedef boost::asio::ip::tcp::resolver asio_resolver_t;
};

template <typename Handler>
struct resolver_adapter_handle_resolve_ep
  : public resolver_adapter_traits
{
  Handler h_;
  std::shared_ptr<asio_resolver_t> r_;

  typedef void result_type;

  resolver_adapter_handle_resolve_ep(Handler h,
      std::shared_ptr<asio_resolver_t> r)
    : h_(h)
    , r_(r)
  {}

  void operator()(const boost::system::error_code& ec,
      typename asio_resolver_t::iterator it)
  {
    basic_resolver_iterator v(it);
    iterator it_adapted = { v };
    return h_(ec, it_adapted);
  }
};

template <typename Handler>
struct resolver_adapter_handle_resolve_aaaa
  : public resolver_adapter_traits
{
  Handler h_;
  query q_;
  int port_;

  typedef void result_type;

  resolver_adapter_handle_resolve_aaaa(Handler h, query q, int port)
    : h_(h)
    , q_(q)
    , port_(port)
  {}

  void operator()(const boost::system::error_code& ec,
      typename resolver::iterator_aaaa it)
  {
    basic_resolver_iterator v(it);
    endpoint_type ep(boost::asio::ip::tcp::v6(), port_);
    iterator it_adapted = { v, q_.host_name(), q_.service_name(), ep };
    h_(ec, it_adapted);
  }
};

template <typename Handler>
struct resolver_adapter_handle_resolve_a
  : public resolver_adapter_traits
{
  Handler h_;
  query q_;
  int port_;

  typedef void result_type;

  resolver_adapter_handle_resolve_a(Handler h, query q, int port)
    : h_(h)
    , q_(q)
    , port_(port)
  {}

  void operator()(const boost::system::error_code& ec,
      typename resolver::iterator_a it)
  {
    basic_resolver_iterator v(it);
    endpoint_type ep(boost::asio::ip::tcp::v4(), port_);
    iterator it_adapted = { v, q_.host_name(), q_.service_name(), ep };
    h_(ec, it_adapted);
  }
};

template <typename Handler>
struct resolver_adapter_handle_resolve
  : public resolver_adapter_traits
{
  Handler h_;
  query q_;
  std::shared_ptr<asio_resolver_t> r_;

  resolver_adapter_handle_resolve(Handler h, query q,
      std::shared_ptr<asio_resolver_t> r)
    : h_(h)
    , q_(q)
    , r_(r)
  {}

  typedef void result_type;

  void operator()(const boost::system::error_code& ec,
      typename asio_resolver_t::iterator it)
  {
    basic_resolver_iterator v(it);
    iterator it_adapted = { v, q_.host_name(), q_.service_name() };
    h_(ec, it_adapted);
  }
};

template <typename Handler>
struct resolver_adapter_handle_resolve_ptr
  : public resolver_adapter_traits
{
  Handler h_;
  endpoint_type e_;

  typedef void result_type;

  resolver_adapter_handle_resolve_ptr(Handler h, endpoint_type e)
    : h_(h)
    , e_(e)
  {
  }

  void operator()(const boost::system::error_code& ec,
      typename resolver::iterator_ptr it)
  {
    basic_resolver_iterator v(it);
    iterator it_adapted = { v, "", "", e_ };
    return h_(ec, it_adapted);
  }
};

class resolver_adapter
  : public resolver_adapter_traits
{
  resolver r_;

public:
  resolver_adapter(boost::asio::io_service& ios)
    : r_(ios)
  {}

  void cancel()
  {
    r_.cancel();
  }

  template <typename Handler>
  void async_resolve(query q, Handler h)
  {
    int port = -1;
    try { port = std::stoi(q.service_name()); }
    catch (...) {}

    // Fallback to asio if service name is not numeric or hostname is empty
    if (port < 0 || q.host_name().empty())
    {
      boost::asio::io_service& ios = r_.get_io_service();
      std::shared_ptr<asio_resolver_t> r(new asio_resolver_t(ios));
      resolver_adapter_handle_resolve<Handler> handler(h, q, r);
      return r->async_resolve(q, handler);
    }

    if (q.hints().ai_family == AF_INET6)
    {
      resolver_adapter_handle_resolve_aaaa<Handler> handler(h, q, port);
      return r_.async_resolve_aaaa(q.host_name(), handler);
    }
    else
    {
      resolver_adapter_handle_resolve_a<Handler> handler(h, q, port);
      return r_.async_resolve_a(q.host_name(), handler);
    }
  }

  template <typename Handler>
  void async_resolve(endpoint_type e, Handler h)
  {
    boost::asio::ip::address addr (e.address());
    // Fallback to asio if address is unspecified
    if (addr.is_unspecified())
    {
      boost::asio::io_service& ios = r_.get_io_service();
      std::shared_ptr<asio_resolver_t> r(new asio_resolver_t(ios));
      resolver_adapter_handle_resolve_ep<Handler> handler(h, r);
      return r->async_resolve(e, handler);
    }
    else
    {
      resolver_adapter_handle_resolve_ptr<Handler> handler(h, e);
      if (addr.is_v4())
        return r_.async_resolve_ptr(rev_order_av4_str(addr.to_v4()), handler);
      else
        return r_.async_resolve_ptr(rev_order_av6_str(addr.to_v6()), handler);
    }
  }
};

#endif
