#ifndef _ADNS_RESOLVER_SERVICE_IPP_
#define _ADNS_RESOLVER_SERVICE_IPP_

#include <adns.h>
#include <poll.h>
#include <boost/iterator/iterator_facade.hpp>
#include <boost/intrusive/slist.hpp>
#include <boost/enable_shared_from_this.hpp>
#include <boost/make_shared.hpp>
#include <boost/bind.hpp>

template <class D, class I, class V>
class adns_resolver_service::basic_iterator
  : public boost::iterator_facade<
    D,
    V,
    boost::forward_traversal_tag,
    V>
{
public:
  basic_iterator()
    : q_()
    , a_()
    , iter_(0)
    , end_(0)
  {
  }

  static D create(basic_query_ptr_t q, adns_answer* a)
  {
    D it;
    it.q_ = q;
    it.a_ = a;
    it.iter_ = a ? it.get_begin(a) : 0;
    it.end_ = it.iter_ + (a ? a->nrrs : 0);
    return it;
  }

private:
  friend class boost::iterator_core_access;

  void increment()
  {
    if (++iter_ == end_)
      *this = basic_iterator<D, I, V>();
  }

  bool equal(const basic_iterator<D, I, V>& other) const
  {
    return iter_ == other.iter_ &&
      end_ == other.end_;
  }

  V dereference() const
  {
    const D* that = static_cast<const D*>(this);
    return that->do_dereference(iter_);
  }

  basic_query_ptr_t q_;
  adns_answer* a_;
  I iter_;
  I end_;
};

class adns_resolver_service::iterator_a
  : public adns_resolver_service::basic_iterator<
    adns_resolver_service::iterator_a, in_addr*, std::string>
{
public:
  in_addr* get_begin(adns_answer* a) const
  {
    return a->rrs.inaddr;
  }

  std::string do_dereference(in_addr* it) const
  {
    const char* rv = inet_ntoa(*it);
    return std::string(rv ? rv : "");
  }
};

class adns_resolver_service::iterator_aaaa
  : public adns_resolver_service::basic_iterator<
    adns_resolver_service::iterator_aaaa, in6_addr*, std::string>
{
public:
  in6_addr* get_begin(adns_answer* a) const
  {
    return a->rrs.in6addr;
  }

  std::string do_dereference(in6_addr* it) const
  {
    char ipv6[40];
    const char* rv = inet_ntop(AF_INET6, &*it, ipv6, sizeof ipv6);
    return std::string(rv ? rv : "");
  }
};

class adns_resolver_service::iterator_ptr
  : public adns_resolver_service::basic_iterator<
    adns_resolver_service::iterator_ptr, char**, std::string>
{
public:
  char** get_begin(adns_answer* a) const
  {
    return a->rrs.str;
  }

  std::string do_dereference(char** it) const
  {
    return std::string(*it);
  }
};

class adns_resolver_service::iterator_txt
  : public adns_resolver_service::basic_iterator<
    adns_resolver_service::iterator_txt, adns_rr_intstr**, std::string>
{
public:
  adns_rr_intstr** get_begin(adns_answer* a) const
  {
    return a->rrs.manyistr;
  }

  std::string do_dereference(adns_rr_intstr** it) const
  {
    return std::string((*it)->str);
  }
};

class adns_resolver_service::iterator_mx
  : public adns_resolver_service::basic_iterator<
    adns_resolver_service::iterator_mx, adns_rr_inthostaddr*, std::string>
{
public:
  adns_rr_inthostaddr* get_begin(adns_answer* a) const
  {
    return a->rrs.inthostaddr;
  }

  std::string do_dereference(adns_rr_inthostaddr* it) const
  {
    return std::string(it->ha.host);
  }
};

typedef boost::intrusive::slist_base_hook<
    boost::intrusive::link_mode<boost::intrusive::auto_unlink>
    > auto_unlink_hook;

struct adns_resolver_service::basic_query
  : public boost::enable_shared_from_this<adns_resolver_service::basic_query>
  , public auto_unlink_hook
{
  enum { n_of_descr = ADNS_POLLFDS_RECOMMENDED };

  adns_state st_;
  stream_descriptor_ptr_t sds_[n_of_descr];
  pollfd fds_[n_of_descr];
  adns_query qu_;
  adns_answer* a_;
  boost::asio::deadline_timer t_;
  implementation_type& d_;
  bool done_;

  explicit basic_query(implementation_type& d);

  virtual ~basic_query()
  {
    if (st_)
      adns_finish(st_);
    if (a_)
      free(a_);
  }

  void cancel()
  {
    adns_finish(st_);
    st_ = 0;
    t_.cancel();
    reset_fds();
    invoke_handler_helper(boost::asio::error::operation_aborted);
  }

  inline void start();
  inline void start_helper();

  inline void reset_fds()
  {
    for (int i=0; i<n_of_descr; ++i)
    {
      stream_descriptor_ptr_t& sd = sds_[i];
      if (sd)
      {
        sd->release();
        sd.reset();
      }
    }
  }

  void before_poll();
  void after_poll();

  void after_poll_helper(const boost::system::error_code& ec,
      std::size_t size)
  {
    if (ec == boost::asio::error::operation_aborted || done_)
      return;

    after_poll();
  }

  virtual void invoke_handler(const boost::system::error_code& ec,
      adns_answer* a) = 0;

  inline void invoke_handler_helper(const boost::system::error_code& ec,
      adns_answer* a = 0);
};

template <class Handler, class Iterator>
struct adns_resolver_service::query
  : public adns_resolver_service::basic_query
{
  Handler handler_;

  query(implementation_type& impl, Handler handler)
    : basic_query(impl)
    , handler_(handler)
  {}

  void invoke_handler(const boost::system::error_code& ec, adns_answer* a)
  {
    handler_(ec, Iterator::create(shared_from_this(), a));
  }
};

struct adns_resolver_service::helper
{
  typedef boost::intrusive::slist<
      basic_query,
      boost::intrusive::constant_time_size<false> > basic_query_list_t;

  basic_query_list_t l_;
  boost::asio::io_service::strand strand_;

  helper(boost::asio::io_service& ios)
    : strand_(ios)
  {}

  void cancel()
  {
    while (!l_.empty())
    {
      basic_query_list_t::iterator it = l_.begin();
      it->cancel();
    }
  }
};

adns_resolver_service::adns_resolver_service(boost::asio::io_service& owner)
  : service_base<adns_resolver_service>(owner)
{
}

void adns_resolver_service::shutdown_service()
{
}

void adns_resolver_service::destroy(implementation_type& impl)
{
  impl.reset();
}

void adns_resolver_service::set_options(
    implementation_type& impl,
    const resolver_options& options)
{
  static_assert(false, "Not implemented");
}

void adns_resolver_service::construct(implementation_type& impl)
{
  impl.reset(new helper(get_io_service()));
}

void adns_resolver_service::cancel(implementation_type& impl)
{
  if (impl)
    impl->strand_.dispatch(boost::bind(&helper::cancel, impl));
}

void adns_resolver_service::basic_query::start()
{
  d_->strand_.dispatch(
    boost::bind(&basic_query::start_helper, shared_from_this()));
}

void adns_resolver_service::basic_query::start_helper()
{
  d_->l_.push_front(*this);
  before_poll();
}

void adns_resolver_service::basic_query::invoke_handler_helper(
  const boost::system::error_code& ec, adns_answer* a)
{
  done_ = true;
  unlink();

  d_->strand_.get_io_service().post(
    boost::bind(&basic_query::invoke_handler,
        shared_from_this(), ec, a));
}

template<typename Handler>
void adns_resolver_service::async_resolve_a(implementation_type& impl,
    const std::string& domain, Handler handler)
{
  typedef query<Handler, iterator_a> query_t;
  basic_query_ptr_t q(new query_t(impl, handler));
  adns_queryflags opt = adns_queryflags(
    adns_qf_quoteok_query |
    adns_qf_quoteok_cname |
    adns_qf_quoteok_anshost |
    adns_qf_cname_loose);
  int rv = adns_submit(q->st_, domain.c_str(), adns_r_a,
      opt, 0, &q->qu_);
  if (rv)
  {
    return handler(
      boost::asio::error::host_not_found_try_again, iterator_a());
  }
  q->start();
}

template<typename Handler>
void adns_resolver_service::async_resolve_aaaa(implementation_type& impl,
    const std::string& domain, Handler handler)
{
  typedef query<Handler, iterator_aaaa> query_t;
  basic_query_ptr_t q(new query_t(impl, handler));
  adns_queryflags opt = adns_queryflags(
    adns_qf_quoteok_query |
    adns_qf_quoteok_cname |
    adns_qf_quoteok_anshost |
    adns_qf_cname_loose);
  int rv = adns_submit(q->st_, domain.c_str(), adns_r_aaaa,
      opt, 0, &q->qu_);
  if (rv)
  {
    return handler(
      boost::asio::error::host_not_found_try_again, iterator_aaaa());
  }
  q->start();
}

template<typename Handler>
void adns_resolver_service::async_resolve_ptr(implementation_type& impl,
    const std::string& ip, Handler handler)
{
  typedef query<Handler, iterator_ptr> query_t;
  basic_query_ptr_t q(new query_t(impl, handler));
  adns_queryflags opt = adns_queryflags(
    adns_qf_quoteok_query |
    adns_qf_quoteok_cname |
    adns_qf_quoteok_anshost |
    adns_qf_cname_loose);
  int rv = adns_submit(q->st_, ip.c_str(), adns_r_ptr_raw,
      opt, 0, &q->qu_);
  if (rv)
  {
    return handler(
      boost::asio::error::host_not_found_try_again, iterator_ptr());
  }
  q->start();
}

template<typename Handler>
void adns_resolver_service::async_resolve_mx(implementation_type& impl,
    const std::string& domain, Handler handler)
{
  typedef query<Handler, iterator_mx> query_t;
  basic_query_ptr_t q(new query_t(impl, handler));
  adns_queryflags opt = adns_queryflags(
    adns_qf_quoteok_query |
    adns_qf_quoteok_cname |
    adns_qf_quoteok_anshost |
    adns_qf_cname_loose);
  int rv = adns_submit(q->st_, domain.c_str(), adns_r_mx,
      opt, 0, &q->qu_);
  if (rv)
  {
    return handler(
      boost::asio::error::host_not_found_try_again, iterator_mx());
  }
  q->start();
}

template<typename Handler>
void adns_resolver_service::async_resolve_txt(implementation_type& impl,
    const std::string& domain, Handler handler)
{
  typedef query<Handler, iterator_txt> query_t;
  basic_query_ptr_t q(new query_t(impl, handler));
  adns_queryflags opt = adns_queryflags(
    adns_qf_quoteok_query |
    adns_qf_quoteok_cname |
    adns_qf_quoteok_anshost |
    adns_qf_cname_loose);
  int rv = adns_submit(q->st_, domain.c_str(), adns_r_txt,
      opt, 0, &q->qu_);
  if (rv)
  {
    return handler(
      boost::asio::error::host_not_found_try_again, iterator_txt());
  }
  q->start();
}

inline boost::system::error_code translate_adns_status(adns_status error)
{
  switch (error)
  {
    case adns_s_ok:
      return boost::system::error_code();
    case adns_s_nomemory:
      return boost::asio::error::no_memory;
    case adns_s_querydomainwrong:
    case adns_s_querydomaininvalid:
    case adns_s_querydomaintoolong:
      return boost::asio::error::no_recovery;
    case adns_s_nxdomain:
    case adns_s_nodata:
      return boost::asio::error::host_not_found;
    default:
      return boost::asio::error::host_not_found_try_again;
  }
}

adns_resolver_service::basic_query::basic_query(implementation_type& d)
  : st_()
  , sds_()
  , fds_()
  , qu_()
  , a_(0)
  , t_(d->strand_.get_io_service())
  , d_(d)
  , done_(false)
{
  adns_initflags opt = adns_initflags(
    adns_if_noautosys | adns_if_noenv | adns_if_noerrprint);

  int rv = adns_init(&st_, opt, 0);
  if (rv || !st_)
  {
    throw std::runtime_error("adns_init");
  }
}

void adns_resolver_service::basic_query::before_poll()
{
  int nfds = n_of_descr;
  int rv = adns_beforepoll(st_, fds_, &nfds, 0, 0);

  timeval* tv_mod  = 0;
  timeval tv_buf, now;
  gettimeofday(&now, 0);
  adns_firsttimeout(st_, &tv_mod, &tv_buf, now);

  boost::asio::io_service& ios = d_->strand_.get_io_service();
  for (int i=0; i<nfds; ++i)
  {
    sds_[i].reset(new stream_descriptor_t(ios,
            fds_[i].fd));
    sds_[i]->async_read_some(boost::asio::null_buffers(),
        d_->strand_.wrap(
          boost::bind(&basic_query::after_poll_helper,
              shared_from_this(), _1, _2)));
  }

  rv = adns_processany(st_);
  if (rv)
  {
    return invoke_handler_helper(boost::asio::error::host_not_found_try_again);
  }

  if (tv_mod == 0)
  {
    after_poll();
  }
  else
  {
    // Set the timer
    t_.expires_from_now(boost::posix_time::seconds(tv_buf.tv_sec)
        + boost::posix_time::microseconds(tv_buf.tv_usec));

    t_.async_wait(d_->strand_.wrap(
          boost::bind(&basic_query::after_poll_helper,
              shared_from_this(), _1, 0)));
  }
}

void adns_resolver_service::basic_query::after_poll()
{
  int nfds = n_of_descr;
  adns_afterpoll(st_, fds_, nfds, 0);
  int rv = adns_processany(st_);
  if (rv)
  {
    return invoke_handler_helper(boost::asio::error::host_not_found_try_again);
  }

  void* context_r = 0;
  if (a_)
  {
    free(a_);
    a_ = 0;
  }
  rv = adns_check(st_, &qu_, &a_, &context_r);

  reset_fds();
  t_.cancel();

  if (!rv)
  {
    return invoke_handler_helper(translate_adns_status(a_->status), a_);
  }
  else
  {
    before_poll();
  }
}

#endif /* _ADNS_RESOLVER_SERVICE_IPP_ */
