from calendar_attach_processor.calendar_attach_processor import CalendarMailhookProcessor
from logbroker_processors.processors import Processor
from logbroker_processors.utils.environment import get_environment_name

from enum import Enum

import logging

logger = logging.getLogger(__name__)


class FolderId(Enum):
    """https://a.yandex-team.ru/arc/trunk/arcadia/mail/macs/include/macs/data/symbols.h?rev=4087216#L9-25"""
    SYM_ZOMBIE_FOLDER = -1  # artificial label for seen(!new) messages
    SYM_FOLDER_NONE = 0  # non-localized folder name
    SYM_INBOX = 1  # folder is INBOX
    SYM_SENT = 2  # folder is SENT
    SYM_TRASH = 3  # folder is TRASH
    SYM_SPAM = 4  # folder is SPAM
    SYM_DRAFTS = 5  # folder is DRAFTS
    SYM_OUTBOX = 6  # folder is OUTBOX
    SYM_ARCHIVE = 7  # folder is ARCHIVE
    SYM_TEMPLATE = 8  # folder is TEMPLATE
    SYM_DISCOUNT = 9  # folder for discount messages
    SYM_UNSUBSCRIBE = 10  # folder for filtered messages(unsubscribe)


IGNORED_FOLDERS = (FolderId.SYM_SPAM.value, FolderId.SYM_TRASH.value)


class MailAttachLogProcessor(Processor):
    CALENDAR_TYPE = ('calendar', 'ics')
    REQ_ARGS = ('tskv_format', 'uid', 'mid', 'stid', 'hid', 'fileType', 'fidSpecType')

    def __init__(self, **kwargs):
        """
        Works with any prod/corp env which is defined with help of tskv_format log field (MPROTO-3998)
        :param kwargs: example in configs/calendar-mailhook/development.json
        """
        super(MailAttachLogProcessor, self).__init__(**kwargs)
        env = get_environment_name()
        self.config = kwargs
        self.processors = {
            # CAL-10712
            # 'mail-mxback-attach-log': CalendarMailhookProcessor.from_config(self.config['prod'], 'public', env)
            #                           if env else NoopCalendarAttachProcessor(),
            # 'mail-mxfront-attach-log': CalendarMailhookProcessor.from_config(self.config['prod'], 'public', env)
            #                            if env else NoopCalendarAttachProcessor(),
            'mail-mxback-attach-log': NoopCalendarAttachProcessor(),
            'mail-mxfront-attach-log': NoopCalendarAttachProcessor(),
            'mail-mxcorp-attach-log': CalendarMailhookProcessor.from_config(self.config['corp'], 'yt', env)
                                      if env else NoopCalendarAttachProcessor(),
            'mail-mxbackcorp-attach-log': CalendarMailhookProcessor.from_config(self.config['corp'], 'yt', env)
                                          if env else NoopCalendarAttachProcessor(),
        }

    @staticmethod
    def is_calendar_type(file_type):
        sub_type = file_type.split('/')[1].lower()
        return sub_type in MailAttachLogProcessor.CALENDAR_TYPE

    def process(self, header, data):
        if all(req_args in data.iterkeys() for req_args in self.REQ_ARGS) \
                and self.is_calendar_type(data['fileType']) \
                and data['tskv_format'] in self.processors:

            logger.debug('action=preprocess\tuid=%s\tmid=%s\tstid=%s\thid=%s\tfileType=%s\tfidSpecType=%s' %
                         (data['uid'], data['mid'], data['stid'], data['hid'], data['fileType'], data['fidSpecType']))

            if int(data['fidSpecType']) not in IGNORED_FOLDERS:
                logger.debug('action=processing\tuid=%s\tmid=%s\tstid=%s\thid=%s\tfileType=%s\tfidSpecType=%s' %
                             (data['uid'], data['mid'], data['stid'], data['hid'], data['fileType'], data['fidSpecType']))
                return self.processors[data['tskv_format']].process_calendar_attach(data['uid'],
                                                                                    data['mid'],
                                                                                    data['stid'],
                                                                                    data['hid'],
                                                                                    data['fileType'])
        return True


class NoopCalendarAttachProcessor:
    def __init__(self, *args, **kwargs):
        pass

    def process_calendar_attach(self, *args):
        return True
