"""
Defines statistics class
"""
import json
import logging
import time
import os
from datetime import datetime as dt


class ClickhouseStats(object):
    """
    Provides common methods and storage for processing statistics.
    """
    def __init__(self):
        self.all = self._init_stat()
        self._log = logging.getLogger('stats.clickhouse')

    def _init_stat(self):
        """
        Defines initial values for statistics.
        """
        return {
            'total': {
                'dropped': 0,
                'read': 0,
                'unknown': 0,
            },
        }

    def update(self, topic, key, value):
        try:
            self.all[topic][key] = value
        except KeyError:
            self.all[topic] = {key: value}

    def get(self, topic, key, default=None):
        try:
            return self.all[topic][key]
        except KeyError:
            return default

    def set(self, topic, key, value):
        return self.update(topic, key, value)

    def incr(self, topic, key, step=1):
        self.update(
            topic,
            key,
            self.get(topic, key, default=0) + step,
        )
        if topic != 'total':
            self.update(
                'total',
                key,
                self.get('total', key, default=0) + step,
            )

    def decr(self, topic, key, step=1):
        return self.incr(topic, key, step * -1)

    def reset(self):
        self.all = self._init_stat()

    def write_stats_log(self):
        self._log.info(self.json())

    def json(self, header='STATS'):
        return json.dumps({
            header: self.all,
            'timestamp': int(time.time()),
            'pid': os.getpid(),
            'date': str(dt.now()),
        })
