from .base import BaseProcessor
import logging
import time

log = logging.getLogger(__name__)


class Nwsmtp(BaseProcessor):

    def nwsmtp_smtpsession_ts(self, header, data):
        """
            Requires smtp-session and timestamp
            """
        assert isinstance(data, dict)

        try:
            smtp_session = data['smtp-session'].strip()
            ts = time.strptime(data['timestamp'], '%Y-%m-%d %H:%M:%S')
        except KeyError:
            raise ValueError("unable to get smtp-session or date from string")
        if not smtp_session:
            raise ValueError("got an empty smtp-session, skipping the line")

        key = "{ssid}_{r_ts:020d}".format(
            ssid=self.md5(smtp_session),
            r_ts=self.reverse_ts(ts)
            )
        processed_data = self.filter_dict(data, ['tskv_format'])

        return {
            'data': processed_data,
            'key': key,
            }

    def nwsmtp_uid_ts(self, header, data):
        """
            Requires smtp-session and timestamp
            """
        assert isinstance(data, dict)

        try:
            smtp_session = data['smtp-session'].strip()
            uid = data['uid'].strip()
            ts = time.strptime(data['timestamp'], '%Y-%m-%d %H:%M:%S')
        except KeyError:
            raise ValueError("unable to get smtp-session or date from string")
        if not uid:
            raise ValueError("got an empty uid, skipping the line")

        key = "{uid}_{r_ts:020d}".format(
            uid=self.md5(uid),
            r_ts=self.reverse_ts(ts)
            )
        processed_data = {
            'ts': str(time.mktime(ts)),
            'uid': uid,
            'message-id': data.get('message-id', ''),
            'smtp-session': smtp_session
            }

        return {
            'data': processed_data,
            'key': key,
            }

    def nwsmtp_from_ts(self, header, data):
        """
            Requires smtp-session, from and timestamp
            """
        assert isinstance(data, dict)

        try:
            sender = data['from'].strip().lower()
            ts = time.strptime(data['timestamp'], '%Y-%m-%d %H:%M:%S')
            smtp_session = data['smtp-session'].strip()
        except (KeyError, ValueError):
            raise ValueError("unable to get sender or session from string")
        if not all([sender, smtp_session]):
            raise ValueError("got an empty localpart or domain, skipping the line")

        key = "{sender}_{r_ts:020d}".format(
            sender=sender,
            r_ts=self.reverse_ts(ts)
            )
        processed_data = {
            'ts': str(time.mktime(ts)),
            'from': sender,
            'smtp-session': smtp_session,
            }

        return {
            'data': processed_data,
            'key': key,
            }
