import logging
import time
from .base import BaseProcessor

log = logging.getLogger(__name__)


class Xiva(BaseProcessor):

    def xiva_events(self, header, data):
        """
            Requires uid, group and timestamp
            Writes to two tables:
              - xiva_subscriptions_uid_ts, if group=subscriptions or group=channel
              - xiva_notifications_uid_ts, if group=notifications
            Anything else is dropped.
            """

        assert isinstance(data, dict)

        table = None
        try:
            ts = time.strptime(data['timestamp'], '%Y-%m-%d %H:%M:%S')
            group = data['group'].strip().lower()
            uid = data['uid'].strip()
        except Exception:
            raise ValueError("unable to get uid or timestamp from string")
        if not uid or not group:
            raise ValueError("got an empty uid or group, skipping the line")

        key = "{uid}_{r_ts:020d}".format(
            uid=self.md5(uid),
            r_ts=self.reverse_ts(ts)
            )

        # select table based on value of "group"
        if group == 'subscriptions' or group == 'channel':
            table = 'xiva_subscriptions_uid_ts'
        elif group == 'notifications':
            table = 'xiva_notifications_uid_ts'
        else:
            raise ValueError("got unexpected group value {g}".format(g=group))

        processed_data = self.filter_dict(data, ['tskv_format'])

        # XXX: The table is set explicitly
        return {
            'data': processed_data,
            'table': table,
            'key': key,
            }
