# encoding: utf-8
from __future__ import unicode_literals

import logging
import os

from collections import deque
from functools import partial
from multiprocessing.dummy import Pool

from logbroker_processors.processors import Processor
from logbroker_processors.utils.service_base_client import ServiceException

from ..utils import passport

from .parser import AbookUserJournalParser
from .client import CollieClient, SenderClient

logger = logging.getLogger(__name__)


class SharpeiProcessor(Processor):
    """Процессор обработки событий регистрации пользователя в Почте, отправляем welcome письмо"""
    def __init__(self, *args, **kwargs):
        super(SharpeiProcessor, self).__init__(*args, **kwargs)

        self._sender = SenderClient(host=kwargs['host'],
                                    timeout=kwargs['timeout'],
                                    account=kwargs['account'],
                                    token=kwargs.get('token', os.environ['SENDER_OAUTH_TOKEN']))
        self._welcome_campaign_slug = kwargs['welcome_campaign_slug']
        self._welcome_campaign_slug_en = kwargs['welcome_campaign_slug_en']
        self._buffer = deque()

    def process(self, header, data):
        try:
            if data.get('operation') == 'registration':
                uid = int(data.get('uid'))
                # Этому пользователю надо слать письмо
                needs_welcome = bool(int(data.get('needsWelcome')))
                # Покладка в шарпее включена
                enable_welcome = bool(int(data.get('enableWelcome')))

                lang = data.get('lang', 'ru')

                if uid and (needs_welcome and not enable_welcome):
                    self._buffer.append((uid, lang))

        except ValueError:
            logger.error('ValueError, skip: %s', data)

        return True

    def flush(self, force=False):
        while self._buffer:
            uid, lang = self._buffer[0]
            slug = self._welcome_campaign_slug_en
            if lang in ('ru', 'uk'):
                slug = self._welcome_campaign_slug

            self._sender.send_email(slug, uid)
            logger.info('Welcome letter send to uid:%s, lang:%s', uid, lang)
            self._buffer.popleft()

        return True


class AbookProcessor(Processor):
    """Процессор UserJornal для сбора контактов из писем и передачи в Abook"""
    MAX_BUFFER_SIZE = 500
    CORP_MAX_BUFFER_SIZE = 10
    MAX_POOL_SIZE = 20
    SENT_F_TYPE = 2

    def __init__(self, *args, **kwargs):
        super(AbookProcessor, self).__init__(*args, **kwargs)
        self._parser = AbookUserJournalParser()

        self._client = CollieClient(**kwargs['collie'])
        self._corp_client = CollieClient(**kwargs['collie-corp'])

        self._buffer = {}
        self._buffer_corp = {}

    def process(self, header, data):
        all_records = self._parser.parse(data)
        if not all_records:
            return False

        valid_records = [rec for rec in all_records if rec and rec.valid]
        if not valid_records:
            return False

        for rec in valid_records:
            if rec.f_type != self.SENT_F_TYPE:
                continue

            if passport.user_is_corp(rec.uid):
                # Если team пользователь
                self._fill_user_buffer(rec, self._buffer_corp)
            else:
                self._fill_user_buffer(rec, self._buffer)

        return True

    def flush(self, force=False):
        if force and self._buffer or len(self._buffer) >= self.MAX_BUFFER_SIZE:
            logger.debug('Flushing user %d records to abook', len(self._buffer))
            self._send_to_abook(self._client, self._buffer)

        if force and self._buffer_corp or len(self._buffer_corp) >= self.CORP_MAX_BUFFER_SIZE:
            logger.debug('Flushing user %d records to abook-corp', len(self._buffer_corp))
            self._send_to_abook(self._corp_client, self._buffer_corp)

    @staticmethod
    def _fill_user_buffer(rec, c_buffer):
        if rec.uid not in c_buffer:
            c_buffer[rec.uid] = {'to': [], 'cc': [], 'bcc': []}

        for header in c_buffer[rec.uid]:
            recipients = getattr(rec, 'email_%s' % header, None)
            if recipients:
                c_buffer[rec.uid][header].append(recipients)

    @staticmethod
    def _send_to_abook_single_uid(client, c_buffer, uid):
        data = c_buffer.get(uid)
        if data is not None:
            try:
                client.feed_uid(uid, data)
                c_buffer.pop(uid, None)
            except ServiceException as e:
                logger.error('Failed to send contacts to %s: %s', client.host, e)
            except:
                logger.exception('Unhandled send abook error')

    def _send_to_abook(self, client, c_buffer):
        pool = Pool(self.MAX_POOL_SIZE)
        try:
            pool.map(partial(self._send_to_abook_single_uid, client, c_buffer), c_buffer.keys())
        finally:
            pool.close()
            pool.join()
            c_buffer.clear()
