# encoding: utf-8
from __future__ import unicode_literals

from datetime import datetime
from hashlib import md5

from .utils import quote_tskv_value, unquote_tskv_value, unquote_unicode_value, parse_state


class BaseUserJournalRecord(object):
    _yandex_mail = '1'
    _source = 'userjournal'

    def __init__(self, uid=None, mid=None, unixtime=None, iso_eventtime='', operation=None, tid=None, fid=None, **kwargs):
        self._valid = True
        if not uid or not unixtime or not operation or not mid:
            self._valid = False
            return
        try:
            action_datetime = datetime.fromtimestamp(float(unixtime))
            self.event_date = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
            self.date = action_datetime.strftime('%Y-%m-%d')
            self.uid = int(uid)
            self.mid = int(mid)
            self.tid = int(tid) if tid else None
            self.fid = int(fid) if fid else None
            self.action_date = action_datetime.strftime('%Y-%m-%d %H:%M:%S')
            self.operation = operation
        except ValueError:
            self._valid = False
            return

        if self.uid < 0 or self.mid < 0 or (self.tid and self.tid < 0) or (self.fid and self.fid < 0):
            self._valid = False
            return

    def _make_tskv_value(self, name, value):
        if not value:
            return None
        return '{}={}'.format(name, value)

    def _make_quoted_tskv_value(self, name, value):
        if not value:
            return None
        return self._make_tskv_value(name, quote_tskv_value(value))

    @property
    def valid(self):
        return self._valid

    def _to_tskv(self):
        if not self._valid:
            raise ValueError("record is incorrect")
        parts = (
            'tskv\tdate={}\toperation={}\tsource={}\town_mail={}\tuid={}\tmid={}\taction_date={}\tevent_date={}'.format(
                self.date,
                self.operation,
                self._source,
                self._yandex_mail,
                self.uid,
                self.mid,
                self.action_date,
                self.event_date
            ),
            self._make_tskv_value('tid', self.tid),
            self._make_tskv_value('fid', self.fid),
        )
        return '\t'.join((x for x in parts if x))

    @property
    def tskv(self):
        return self._to_tskv() + '\n'


class ReceiveRecord(BaseUserJournalRecord):
    def __init__(self, emailFrom=None, subject=None, msgId=None, state=None,
                 emailTo=None, emailCc=None, emailBcc=None, ftype=None, **kwargs):
        super(ReceiveRecord, self).__init__(**kwargs)
        if not self._valid:
            return

        self.email_from = emailFrom.lower() if emailFrom else None
        self.local_from = None
        self.domain_from = None
        if self.email_from:
            parts = self.email_from.rsplit('@', 1)
            if len(parts) == 2:
                self.local_from, self.domain_from = parts

        self.subject = unquote_unicode_value(subject) if subject else ''
        self.subject_hash = self.__hash(self.subject) if self.subject else None
        self.message_id = msgId
        self.email_to = emailTo
        self.email_cc = emailCc
        self.email_bcc = emailBcc

        try:
            self.f_type = int(ftype)
        except ValueError:
            self._valid = False

        parsed_state = self._parse_state(state)
        parsed_state_lower = {k.lower(): v for k, v in parsed_state.items()}

        self.x_mailer = parsed_state_lower.get('x-mailer', '')
        self.list_id = parsed_state_lower.get('list-id', '')
        self.x_mailru_msgtype = parsed_state_lower.get('x-mailru-msgtype', '')

    @staticmethod
    def _parse_state(state):
        # tskv unquote
        value = unquote_tskv_value(state)
        # split values
        result = parse_state(value)
        return result or {}

    def __hash(self, value):
        return md5(value.encode('utf-8')).hexdigest()

    def _to_tskv(self):
        _parts = (
            super(ReceiveRecord, self)._to_tskv(),
            self._make_tskv_value('email_from', self.email_from),
            self._make_tskv_value('local_from', self.local_from),
            self._make_tskv_value('domain_from', self.domain_from),
            self._make_quoted_tskv_value('subject', self.subject),
            self._make_tskv_value('subject_hash', self.subject_hash),
            self._make_quoted_tskv_value('message_id', self.message_id),
            self._make_quoted_tskv_value('x_mailer', self.x_mailer),
            self._make_quoted_tskv_value('list_id', self.list_id),
            self._make_quoted_tskv_value('x_mailru_msgtype', self.x_mailru_msgtype),
        )
        return '\t'.join((x for x in _parts if x))


class AbuseRecord(BaseUserJournalRecord):
    def __init__(self, abuseType=None, **kwargs):
        if not abuseType:
            self._valid = False
            return
        super(AbuseRecord, self).__init__(**kwargs)
        if not self._valid:
            return
        self.operation += '_' + abuseType


class MarkRecord(BaseUserJournalRecord):
    def __init__(self, msgStatus=None, **kwargs):
        if not msgStatus:
            self._valid = False
            return
        super(MarkRecord, self).__init__(**kwargs)
        if not self._valid:
            return
        self.operation += '_' + msgStatus


class LinkClickRecord(BaseUserJournalRecord):
    def __init__(self, **kwargs):
        super(LinkClickRecord, self).__init__(**kwargs)
        if not self._valid:
            return
        self.operation = 'link_click'


class MoveRecord(BaseUserJournalRecord):
    def __init__(self, destFid=None, **kwargs):
        super(MoveRecord, self).__init__(**kwargs)
        if not self._valid:
            return
        try:
            self.fid = int(destFid) if destFid else None
        except ValueError:
            self._valid = False
            return
        if self.fid and self.fid < 0:
            self._valid = False
            return


class StartReadingRecord(BaseUserJournalRecord):
    def __init__(self, percentDisplayed=None, **kwargs):
        super(StartReadingRecord, self).__init__(**kwargs)
        if not self._valid:
            return
        self.operation = 'start_reading'


class StopReadingRecord(BaseUserJournalRecord):
    def __init__(self, percentDisplayed=None, **kwargs):
        super(StopReadingRecord, self).__init__(**kwargs)
        if not self._valid:
            return
        self.operation = 'stop_reading'
        if percentDisplayed and percentDisplayed.isdigit():
            self.readed_percent = percentDisplayed
        else:
            self.readed_percent = 0

    def _to_tskv(self):
        _parts = (
            super(StopReadingRecord, self)._to_tskv(),
            self._make_tskv_value('readed_percent', self.readed_percent),
        )
        return '\t'.join((x for x in _parts if x))


class ViewRecord(BaseUserJournalRecord):
    def __init__(self, widgetSubType=None, widgetType=None, **kwargs):
        super(ViewRecord, self).__init__(**kwargs)
        self.widget = widgetType or ''
        if widgetSubType:
            self.widget += ':' + widgetSubType

    def _to_tskv(self):
        _parts = (
            super(ViewRecord, self)._to_tskv(),
            self._make_tskv_value('widget', self.widget),
        )
        return '\t'.join((x for x in _parts if x))
