import os
import urlparse
import logging
import time

import requests

from logbroker_processors.utils.tvm import sessions_producer
from utvm import TVM

logger = logging.getLogger()


class ServiceException(Exception):
    pass


class ServiceRepeatRequest(ServiceException):
    pass


class ServiceBadRequest(ServiceException):
    pass


class ServiceBaseClient(object):
    RETRY_TIMEOUT = [0.3, 0.5, 0.7]

    def __init__(self, host, timeout=15, tvm_id=None, tvm_secret=None, tvm_dst=None):
        self.host = str(host).rstrip('/')
        self.timeout = timeout

        if tvm_id and (tvm_secret in os.environ) and tvm_dst:
            self._session = sessions_producer(TVM(tvm_id, os.environ[tvm_secret], use_v1=False))(tvm_dst)
        else:
            self._session = requests.Session()

    @staticmethod
    def _check_response_code(response):
        if response.status_code < 400:
            return
        elif response.status_code < 500:
            raise ServiceBadRequest('status code: %s' % response.status_code)
        else:
            raise ServiceRepeatRequest('status code: %s' % response.status_code)

    def _process_response(self, r):
        return r.json()

    def _make_request(self, url, headers=None, data=None, params=None, json_body=None, method='post'):
        request_url = urlparse.urljoin(self.host, url)
        for i, retry_timeout in enumerate(self.RETRY_TIMEOUT):
            try:
                try:
                    request_method = getattr(self._session, method.lower())
                    r = request_method(request_url,
                                       headers=headers,
                                       timeout=self.timeout,
                                       params=params,
                                       data=data,
                                       json=json_body)
                    self._check_response_code(r)
                    return self._process_response(r)
                except requests.ConnectionError:
                    raise ServiceRepeatRequest
            except ServiceRepeatRequest:
                if i == len(self.RETRY_TIMEOUT) - 1:
                    raise ServiceException('retry timeout exceeded')
                time.sleep(retry_timeout)
            except ServiceBadRequest as e:
                logger.warning(
                    'Failed to send request %s, params=%s, json=%s, error=%s',
                    request_url, params, json_body, e.message
                )
                return {}

    def endpoint_url(self, relative_url, base_url_override=None):
        host = base_url_override or self.host
        return '%s/%s' % (host, relative_url)

    def get(self, url, **kwargs):
        return self._make_request(url, method='GET', **kwargs)

    def post(self, url, **kwargs):
        return self._make_request(url, method='POST', **kwargs)
