import requests
from itertools import chain

from .windat_logger import get_logger

log = get_logger(__name__)

TRIES = 3


def http_get(url, data=None, headers=None, timeout=2):
    url = url.encode('utf-8')
    for _ in range(TRIES):
        try:
            r = (requests.get(url, headers=headers) if data is None
                 else requests.post(url, data))
            if r.status_code in [200, 204, 206]:
                return r.text
        except requests.HTTPError:
            log.info('Retrying http request url=%s data=%s headers=%s' % (url, str(data), str(headers)))
    log.warning('Failed http request: url=%s data=%s headers=%s' % (url, str(data), str(headers)))
    raise requests.HTTPError


def make_bad_symbols():
    return frozenset(chain(
        [u'\u0000'],
        (unichr(i) for i in xrange(ord(u'\ud800'), ord(u'\udbff') + 1)),
        (unichr(i) for i in xrange(ord(u'\udc00'), ord(u'\udfff') + 1)),
    ))

BAD_SYMBOLS = make_bad_symbols()


def sanitize_unicode(u_str):
    # http://unicodebook.readthedocs.org/en/latest/unicode_encodings.html#surrogates
    #
    return u''.join(x for x in u_str if x not in BAD_SYMBOLS)


def safe_unicode(v, field_name):
    if v is None:
        return v
    if isinstance(v, unicode):
        return sanitize_unicode(v)
    try:
        uv = unicode(v, 'utf-8')
    except Exception as exc:
        log.debug("{0}: {1}".format(field_name, exc))
        try:
            uv = unicode(v, 'utf-8', errors='ignore')
        except Exception as exc:
            log.debug('Fallback unicoding string %s=\'%s\' failed with exc: %s',
                      field_name, v, exc)
    return sanitize_unicode(uv)
