# -*- coding: utf-8 -*-
import datetime


class MessageChunk(object):
    def __init__(self, header, data):
        self.topic = header.get('topic', '')
        self.partition = header.get('partition')
        self.offset = header.get('offset')

        self.server = header.get('server', '')
        self.file = header.get('path', '')
        self.data = data


class BatchCollector(object):
    def __init__(self, process_fn, flush_batch_size=50, flush_time_milliseconds=1000, flush_on_add=True):
        self.process_fn = process_fn
        self.flush_batch_size = flush_batch_size
        self.flush_time_milliseconds = flush_time_milliseconds
        self.flush_on_add = flush_on_add
        self.last_flush_time = datetime.datetime.now()
        self.collection = []

    def add(self, items):
        '''
        Adds items to batch and flushes them if boundary values were reached
        :param items: items for batching
        :return: bool is flushed
        '''
        self.collection.extend(items)
        if self.flush_on_add:
            return self.flush()
        else:
            return False

    def _check_flush(self):
        deltatime = datetime.timedelta(milliseconds=self.flush_time_milliseconds)
        if len(self.collection) >= self.flush_batch_size or self.last_flush_time + deltatime <= datetime.datetime.now():
            return True
        return False

    def _flush(self):
        self.process_fn(self.collection)
        self.last_flush_time = datetime.datetime.now()
        self.collection = []

    def flush(self, force=False):
        if force or self._check_flush():
            self._flush()
            return True
        return False
