# -*- coding: utf-8 -*-
import os
import time
import signal
import logging
from collections import namedtuple
from multiprocessing import Process

from ..utils import importobj
from .queues.mpqueue import MPQueue


INTERRUPTED = False
CHECK_CHILD_PROCESSES = False

ProcessHolder = namedtuple('ProcessHolder', 'process instance')

log = logging.getLogger('runner.arbiter')


class Arbiter(object):
    TASKS_QUEUE_SIZE = 10000
    ACK_QUEUE_SIZE = 20000
    WAIT_WORKERS = 60

    SIGNALS = [signal.SIGCHLD, signal.SIGTERM, signal.SIGINT]

    def __init__(self):
        pass

    def run(self, config):
        log.info('Setup arbiter...')
        workers_config = config['workers']
        emitter_config = config['emitter']

        self.workers_count = workers_config['count']
        self.worker_cls = importobj(workers_config['class'])
        self.worker_args = workers_config.get('args', {})

        self.emitter_cls = importobj(emitter_config['class'])
        self.emitter_args = emitter_config.get('args', {})

        tasks_storage_cls = importobj(emitter_config['storage']['class'])
        tasks_storage_args = emitter_config['storage'].get('args', {})

        self.ack_queue = MPQueue(self.ACK_QUEUE_SIZE)
        self.tasks_queue = MPQueue(self.TASKS_QUEUE_SIZE)
        self.tasks_storage = tasks_storage_cls(**tasks_storage_args)
        self.ctl_queue = MPQueue(5)

        self.emitter = None
        self.workers = []

        log.info('Setup signal handlers')
        for signum in self.SIGNALS:
            signal.signal(signum, self.handle_signal)

        log.info('Run processes...')
        try:
            self.run_emitter(self.emitter_args,
                             self.ack_queue,
                             self.tasks_queue,
                             self.tasks_storage)
            self.run_workers(self.worker_args,
                             self.ack_queue,
                             self.tasks_queue)
            self.loop()
        except KeyboardInterrupt:
            log.info('Got KeyboardInterrupt')
        except:
            log.exception('Unhandled error')
        finally:
            self.cleanup()

    def run_emitter(self, emitter_args, ack_queue, tasks_queue, tasks_storage):
        log.info('Build emitter with args: %s', emitter_args)
        emitter = self.emitter_cls(**emitter_args)
        process = Process(target=emitter.run,
                          args=(ack_queue, tasks_queue, tasks_storage))
        emitter.process = process
        self.emitter = ProcessHolder(process, emitter)
        process.start()
        log.info('Emitter PID: %s', emitter.process.pid)

    def run_workers(self, worker_args, ack_queue, tasks_queue):
        for i in xrange(self.workers_count):
            log.info('Build worker [%s] with args: %s', i, worker_args)
            worker = self.worker_cls(**worker_args)
            process = Process(target=worker.run,
                              args=(ack_queue, tasks_queue))
            process_holder = ProcessHolder(process, worker)
            self.workers.append(process_holder)
        for i, worker in enumerate(self.workers):
            worker.process.start()
            log.info('Worker [%s][PID:%s] started', i, worker.process.pid)

    def signal_to_emitter(self, signum):
        log.info('Send signal:%s to emitter[%s]', signum, self.emitter.process.pid)
        self.send_signal(self.emitter.process, signum)

    def signal_to_workers(self, signum):
        for i, worker in enumerate(self.workers):
            log.info('Send signal:%s to worker[%s][PID:%s]', signum, i, worker.process.pid)
            self.send_signal(worker.process, signum)

    def send_signal(self, proc, signum, ignore_error=True):
        try:
            os.kill(proc.pid, signum)
        except OSError:
            if not ignore_error:
                raise

    def is_alive(self, proc):
        try:
            self.send_signal(proc, 0, False)
            return True
        except OSError:
            return False

    def send_signals_to_childs(self, signum):
        log.info('Signal to emitter (%s)...', signum)
        self.signal_to_emitter(signum)
        log.info('Signal to workers (%s)...', signum)
        self.signal_to_workers(signum)

    def get_child_processes(self):
        return [self.emitter.process] + [w.process for w in self.workers]

    def filter_alive(self, processes):
        return [p for p in processes if self.is_alive(p)]

    def send_signal_and_wait_for_stop(self, sig):
        self.send_signals_to_childs(sig)
        log.info('Wait for %s seconds' % self.WAIT_WORKERS)
        start_wait = time.time()
        child_processes = self.get_child_processes()
        while time.time() <= start_wait + self.WAIT_WORKERS and child_processes:
            time.sleep(1)
            child_processes = self.filter_alive(child_processes)
            log.info('Alive: %s', ','.join([str(p.pid) for p in child_processes]))
        return bool(child_processes)

    def cleanup(self):
        for signum in self.SIGNALS:
            signal.signal(signum, signal.SIG_IGN)
        log.info('Cleanup...')
        has_child_processes = self.send_signal_and_wait_for_stop(signal.SIGINT)
        if has_child_processes:
            log.info('SIGINT failed, trying SIGTERM...')
            has_child_processes = self.send_signal_and_wait_for_stop(signal.SIGTERM)
        if has_child_processes:
            log.info('Force kill')
            self.send_signals_to_childs(signal.SIGKILL)
        log.info('Terminated')

    def loop(self):
        log.info('Run arbiter loop')
        global CHECK_CHILD_PROCESSES
        global INTERRUPTED
        while not INTERRUPTED:
            if CHECK_CHILD_PROCESSES:
                CHECK_CHILD_PROCESSES = False
                # сигнал SIGCHLD может прийти не только от убитого дочернего воркера или эмитера, но и от других процессов
                # например, такое происходило при вызове requests, который внутри себят обращался к _syscmd_uname, что
                # порождало дочерний процесс, который завершался и родительский процесс получал SIGCHLD
                processes = self.get_child_processes()
                for process in processes:
                    # нельзя проверять через посылку сигнала 0 дочернему процессу, так как он в это время висит зомбиком
                    # и сигнал проходит, но при этом процесс уже завершен
                    if not process.is_alive():
                        try:
                            process.join(timeout=3)
                        except OSError:
                            pass
                        return

            time.sleep(1)

    def handle_signal(self, signum, frame):
        if signum == signal.SIGCHLD:
            global CHECK_CHILD_PROCESSES
            CHECK_CHILD_PROCESSES = True
        else:
            global INTERRUPTED
            INTERRUPTED = True
