# -*- coding: utf-8 -*-
import time
import logging

from ..queues.exceptions import (
    FullQueueException,
    EmptyQueueException,
)
from .base import Emitter


log = logging.getLogger('runner.emitters.basic')


class BasicEmitter(Emitter):
    ACK_TIMEOUT = 1
    PUT_TIMEOUT = 1
    ACKS_PER_LOOP = 100
    TASK_REPLAY_TIMEOUT = 100

    def handle_readportion(self):
        raise NotImplementedError()

    def handle_ack(self, task_id):
        raise NotImplementedError()

    def handle_cannot_put_task(self, task, dataitem, replay=False):
        raise NotImplementedError()

    def handle_build_task(self, dataitem):
        raise NotImplementedError()

    def put_task(self, task, replay=False):
        try:
            self.tasks_queue.put_timeout(task, self.PUT_TIMEOUT)
            if replay:
                task.set_time(time.time())
            self.tasks_storage.save(task)
            return True
        except FullQueueException:
            log.info('Queue is full: %s', task.id_)
            if not replay:
                self.tasks_storage.delete(task.id_)

    def done_task(self, task_id):
        self.tasks_storage.delete(task_id)

    def handle_acks(self):
        log.info('Handle acks')
        for _ in xrange(self.ACKS_PER_LOOP):
            try:
                task_id = self.ack_queue.get_timeout(self.ACK_TIMEOUT)
                if self.handle_ack(task_id):
                    self.done_task(task_id)
            except EmptyQueueException:
                break

    def replay_tasks(self):
        log.info('Replay tasks')
        now = time.time()
        missing_tasks = self.tasks_storage.old_tasks(
            now - self.TASK_REPLAY_TIMEOUT
        )
        for task in missing_tasks:
            log.info('Replay task: %s', task.id_)
            if not self.put_task(task, replay=True):
                break

    def loop(self):
        dataitems = []
        while True:
            if not dataitems:
                dataitems = self.handle_readportion()
            while dataitems:
                dataitem = dataitems.pop()
                task = self.handle_build_task(dataitem)
                log.info('Created task %s from dataitem: %s', task.id_, dataitem)
                if not self.put_task(task):
                    self.handle_cannot_put_task(task, dataitem)
                    break
            self.handle_acks()
            self.replay_tasks()
