# -*- coding: utf-8 -*-
import os
import types
from importlib import import_module
from functools import wraps
import logging
import time


log = logging.getLogger(__name__)


def find(matcher, top='.'):
    for dirpath, _dirnames, filenames in os.walk(top):
        for f in filenames:
            f = os.path.relpath(os.path.join(dirpath, f), top)
            basename = os.path.basename(f)
            if matcher.match(basename):
                yield f


def importobj(objpath):
    module_path, obj_name = objpath.rsplit('.', 1)
    try:
        module = import_module(module_path)
    except ImportError, e:
        raise ImportError(objpath, str(e))
    return getattr(module, obj_name)


def retriable_n(retry_count=3, time_sleep=0.2, exceptions=(Exception,)):
    def retriable_n_deco(func):
        @wraps(func)
        def wrapper(*args, **kw):
            for i in range(retry_count - 1):
                try:
                    return func(*args, **kw)
                except Exception, e:
                    if isinstance(e, exceptions):
                        log.warning('%s(*%s, **%s) try %i failed, retrying: %s', func.__name__, args, kw, i, e)
                        time.sleep(time_sleep)
                    else:
                        raise
            else:
                return func(*args, **kw)

        return wrapper

    return retriable_n_deco


def backoff_retriable_n(delay_max=30, delay_min=0.5, exceptions=(Exception,)):
    """
    Used on functions that are expected to to fail on regular basis, e.g.
    suggest
    """
    def backoff_retriable_n_deco(func):
        @wraps(func)
        def wrapper(*args, **kw):
            delay = delay_min
            attempt = 1
            while delay <= delay_max:
                try:
                    return func(*args, **kw)
                except Exception, exc:
                    if isinstance(exc, exceptions):
                        log.debug(
                            '%s(*%s, **%s) try %i failed, waiting %is and retrying: %s',
                            func.__name__, args, kw, attempt, delay, exc
                        )
                        time.sleep(delay)
                        delay *= 2
                        attempt += 1
                    else:
                        raise
            else:
                return func(*args, **kw)

        return wrapper

    return backoff_retriable_n_deco


def smart_str(s, encoding='utf-8', strings_only=False, errors='strict'):
    """
    Returns a bytestring version of 's', encoded as specified in 'encoding'.

    If strings_only is True, don't convert (some) non-string-like objects.
    """
    if strings_only and isinstance(s, (types.NoneType, int)):
        return s
    elif not isinstance(s, basestring):
        try:
            return str(s)
        except UnicodeEncodeError:
            if isinstance(s, Exception):
                # An Exception subclass containing non-ASCII data that doesn't
                # know how to print itself properly. We shouldn't raise a
                # further exception.
                return ' '.join([
                    smart_str(arg, encoding, strings_only, errors) for arg in s
                ])
            return unicode(s).encode(encoding, errors)
    elif isinstance(s, unicode):
        return s.encode(encoding, errors)
    elif s and encoding != 'utf-8':
        return s.decode('utf-8', errors).encode(encoding, errors)
    else:
        return s
