package client

import (
	"context"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/config"
)

type ITvmClient interface {
	GetTicket(string) string
}

type TvmClient struct {
	tvmtool.Client
	logger *zap.Logger
}

func (c TvmClient) GetTicket(alias string) string {
	ticket, err := c.GetServiceTicketForAlias(context.TODO(), alias)
	if err != nil {
		retryable := false
		if tvmErr, ok := err.(*tvm.Error); ok {
			retryable = tvmErr.Retriable
		}

		c.logger.Error(
			"Error during getting service ticket",
			log.String("alias", alias),
			log.Error(err),
			log.Bool("retryable", retryable),
		)
	}
	return ticket
}

func NewTvmClient(cfg *config.Config, logger *zap.Logger) TvmClient {
	if cfg.LB.Token != "" {
		return TvmClient{}
	}
	tvmClient, err := tvmtool.NewClient(cfg.TvmURL, tvmtool.WithAuthToken(cfg.TvmToken), tvmtool.WithLogger(logger))
	if err != nil {
		logger.Fatal("Error on tvmclient create", log.Error(err))
	}
	return TvmClient{
		Client: *tvmClient,
		logger: logger,
	}
}
