package data

import (
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/metrics"
)

type UID int64
type Module string
type Date string
type UserAgent string

type ModuleActivity struct {
	Module Module
	UID    UID
	Date   Date
	Hidden bool
}

const TZ = "Europe/Moscow"

var location, _ = time.LoadLocation(TZ)

var KnownModules = map[string]bool{
	"mobile":        true,
	"mailbox_oper":  true,
	"yserver_pop":   true,
	"sharpei":       true,
	"jsintegration": true,
	"fastsrv":       true,
	"sendbernar":    true,
	"hound":         true,
	"yserver_imap":  true,
	"message_body":  true,
}

func ParseData(data string, yasm *metrics.Yasm) (ModuleActivity, error) {
	var res ModuleActivity
	v := make(map[string]string)
	for _, f := range strings.Split(string(data), "\t") {
		parts := strings.SplitN(f, "=", 2)
		if len(parts) != 2 {
			continue
		}
		v[parts[0]] = parts[1]
	}
	if v["uid"] == "" || v["module"] == "" || (v["date"] == "" && v["unixtime"] == "") {
		yasm.Update("parse_errors", 1)
		return res, errors.New("empty uid, module or date")
	}

	uid, err := strconv.ParseInt(v["uid"], 10, 64)
	if err != nil {
		yasm.Update("parse_errors", 1)
		return res, err
	}

	var t time.Time
	switch {
	case v["date"] != "":
		ts, err := strconv.ParseInt(v["date"], 10, 64)
		if err != nil {
			yasm.Update("parse_errors", 1)
			return res, err
		}
		t = time.Unix(ts/1000, 0)
	default:
		ts, err := strconv.ParseInt(v["unixtime"], 10, 64)
		if err != nil {
			yasm.Update("parse_errors", 1)
			return res, err
		}
		t = time.Unix(ts, 0)
	}

	res.UID = UID(uid)
	res.Date = Date(fmt.Sprintf(
		"%d-%02d-%02d",
		t.In(location).Year(),
		t.In(location).Month(),
		t.In(location).Day(),
	))
	res.Module = Module(v["module"])
	res.Hidden = v["hidden"] == "1"

	yasm.Update("parse_parses", 1)
	yasm.Update(fmt.Sprintf("parse_parses.%s", v["module"]), 1)
	if res.Hidden {
		yasm.Update("parse_hidden", 1)
		yasm.Update(fmt.Sprintf("parse_hidden.%s", v["module"]), 1)
	}

	return res, nil
}
