package lbreader

import (
	"context"
	"crypto/tls"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/log/corelogadapter"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	ydb_tvm "a.yandex-team.ru/library/go/yandex/ydb/auth/tvm2"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/client"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/config"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/metrics"
	"a.yandex-team.ru/market/sre/library/golang/logger"
)

type Reader struct {
	c      persqueue.Reader
	logger *zap.Logger
}

var reader Reader

const LogbrokerTvmAlias = "logbroker"

func (r *Reader) Start(
	cfg *config.Config,
	tvmClient *client.TvmClient,
	logger *zap.Logger,
	yasm *metrics.Yasm,
) (persqueue.Reader, error) {
	r.logger = logger
	var tlsConfig *tls.Config
	if cfg.LB.UseTLS {
		tlsConfig = &tls.Config{}
	}

	creds, err := credentialsProvider(cfg, tvmClient, logger)
	if err != nil {
		return nil, err
	}

	c := persqueue.NewReader(persqueue.ReaderOptions{
		Endpoint:              cfg.LB.Endpoint,
		Credentials:           creds,
		TLSConfig:             tlsConfig,
		Database:              cfg.LB.Database,
		Consumer:              cfg.LB.Consumer,
		Logger:                corelogadapter.New(logger),
		Topics:                []persqueue.TopicInfo{{Topic: cfg.LB.Topic}},
		MaxReadSize:           10 * 1024 * 1024, // 10 mb
		MaxReadMessagesCount:  uint32(cfg.LB.MaxMessages),
		DecompressionDisabled: false,
		RetryOnFailure:        true,
	})

	r.c = c
	ctx := context.Background()
	if _, err := c.Start(ctx); err != nil {
		return c, err
	}

	return c, nil
}

func (r *Reader) Read(
	onMessage func(message string),
	onData func(),
	onClose func(),
	yasm *metrics.Yasm,
) {
	ticker := time.NewTicker(1 * time.Second)
	done := make(chan bool)
	var stat persqueue.Stat
	var memUsage float64
	var bytesRead float64

	go func() {
		for {
			select {
			case <-done:
				return
			case <-ticker.C:
				stat = r.c.Stat()
				memUsage = float64(stat.MemUsage)
				bytesRead = float64(stat.BytesRead)
				yasm.Update("lb_max_memory_usage", memUsage)
				yasm.Update("lb_min_memory_usage", memUsage)
				yasm.Update("lb_bytes_read", bytesRead)
				yasm.Update("lb_in_flight", float64(stat.InflightCount))
				yasm.Update("lb_wait_ack", float64(stat.WaitAckCount))
			}
		}
	}()

	for msg := range r.c.C() {
		switch v := msg.(type) {
		case *persqueue.Data:
			for _, b := range v.Batches() {
				logger.Trace("Received batch",
					log.Any("topic", b.Topic),
					log.Any("partition", b.Partition),
					log.Any("messages", len(b.Messages)))

				for _, m := range b.Messages {
					yasm.Update("lb_messages", 1)
					logger.Trace("Received message", log.Any("seqNo", m.SeqNo))
					onMessage(string(m.Data))
				}
			}
			onData()
			v.Commit()

		default:
		}
	}

	<-r.c.Closed()
	onClose()

	if err := r.c.Err(); err != nil {
		r.logger.Fatal("Reader error", log.Error(err))
	}
}

func (r *Reader) Stop() {
	r.c.Shutdown()
	r.logger.Info("Reader stopped")
}

func credentialsProvider(cfg *config.Config, tvmClient *client.TvmClient, logger *zap.Logger) (ydb.Credentials, error) {
	if cfg.LB.Token != "" {
		return ydb.AuthTokenCredentials{AuthToken: cfg.LB.Token}, nil
	}

	tvmProvider, err := ydb_tvm.NewTvmCredentialsForAlias(context.Background(), tvmClient, LogbrokerTvmAlias, logger)
	if err != nil {
		logger.Error("cannot create tvm provider", log.Error(err))
		return tvmProvider, err
	}
	return tvmProvider, nil
}
