package main

import (
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/client"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/config"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/data"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/lbreader"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/metrics"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/storage"
)

var (
	endpoint      = flag.String("endpoint", "sas.logbroker.yandex.net", "LB endpoint")
	topic         = flag.String("topic", "/userjournal/mail-user-journal-log", "Topic")
	consumer      = flag.String("consumer", "/shared/userjournal", "Consumer")
	database      = flag.String("database", "/Root", "Database")
	useTLS        = flag.Bool("use-tls", false, "Use TLS connection")
	maxMessages   = flag.Int("max-messages", 500, "MaxReadMessagesCount")
	flushMessages = flag.Int("flush-messages", 25, "Flush messages")
	version       = flag.Bool("version", false, "Print version")

	reader lbreader.Reader
	buffer data.Buffer
	logger *zap.Logger
	yasm   *metrics.Yasm
	st     storage.Storage
)

const Version = "1.0.9"

func main() {
	flag.Parse()
	cfg := config.NewConfig(*endpoint, *topic, *consumer, *database, *useTLS, *maxMessages)

	if *version {
		fmt.Printf("Actdb consumer version %s\n", Version)
		os.Exit(0)
	}

	logger = zap.Must(cfg.Logger)

	tvm := client.NewTvmClient(cfg, logger)
	yasm = metrics.NewYasm(logger)
	registerSignals(yasm)

	if _, err := st.Connect(cfg, yasm, logger); err != nil {
		logger.Error("Storage connect error", log.Error(err))
		return
	}
	defer st.Close()

	if _, err := reader.Start(cfg, &tvm, logger, yasm); err != nil {
		logger.Error("Unable to start reader", log.Error(err))
		return
	}

	osChan := make(chan os.Signal, 1)
	signal.Notify(osChan, os.Interrupt, syscall.SIGTERM)
	go func() {
		<-osChan
		logger.Warnf("Interrupt, %d items left in buffer", buffer.Length())
		reader.Stop()
		time.Sleep(500 * time.Millisecond)
		var ok bool
		for !ok {
			ok = buffer.Flush(&st, logger, yasm)
			time.Sleep(250 * time.Millisecond)
		}
		st.Close()
		os.Exit(1)
	}()

	reader.Read(onMessage, onData, func() { buffer.Flush(&st, logger, yasm) }, yasm)
}

func onMessage(msg string) {
	item, err := data.ParseData(msg, yasm)
	switch {
	case err != nil:
		logger.Warn(fmt.Sprintf("Parser error: %s", err), log.Any("data", string(msg)), log.Error(err))

	case item.Module == "fastsrv" || item.Module == "settings":
		logger.Tracef("Parser skipped module %s", item.Module)
	default:
		buffer.Add(&item)
		logger.Trace("Parsed activity", log.Any("item", item))
	}
}

func onData() {
	bufSize := buffer.Length()
	yasm.Update("buffer_size", float64(bufSize))

	if buffer.Length() > *flushMessages {
		buffer.Flush(&st, logger, yasm)
	}
}

func registerSignals(yasm *metrics.Yasm) {
	yasm.RegisterSignal("parse_parses", unistat.NewNumeric("parse_parses", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("parse_errors", unistat.NewNumeric("parse_errors", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("parse_hidden", unistat.NewNumeric("parse_hidden", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("buffer_size", unistat.NewNumeric("buffer_size", 1, aggr.Absolute(), unistat.Last))
	yasm.RegisterSignal("buffer_flushes", unistat.NewNumeric("buffer_flushes", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("buffer_errors", unistat.NewNumeric("buffer_errors", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("db_queries", unistat.NewNumeric("db_queries", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("db_errors", unistat.NewNumeric("db_errors", 1, aggr.Counter(), unistat.Sum))
	for mod := range data.KnownModules {
		yasm.RegisterSignal(fmt.Sprintf("parse_parses.%s", mod), unistat.NewNumeric(fmt.Sprintf("parse_parses.%s", mod), 1, aggr.Counter(), unistat.Sum))
		yasm.RegisterSignal(fmt.Sprintf("parse_hidden.%s", mod), unistat.NewNumeric(fmt.Sprintf("parse_hidden.%s", mod), 1, aggr.Counter(), unistat.Sum))
	}
	yasm.RegisterSignal("lb_max_memory_usage", unistat.NewNumeric("lb_max_memory_usage", 1, metrics.Axxx, unistat.Max))
	yasm.RegisterSignal("lb_min_memory_usage", unistat.NewNumeric("lb_min_memory_usage", 1, metrics.Annn, unistat.Max))
	yasm.RegisterSignal("lb_bytes_read", unistat.NewNumeric("lb_bytes_read", 1, aggr.Counter(), unistat.Max))
	yasm.RegisterSignal("lb_errors", unistat.NewNumeric("lb_errors", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("lb_messages", unistat.NewNumeric("lb_messages", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("lb_in_flight", unistat.NewNumeric("lb_in_flight", 1, aggr.Counter(), unistat.Sum))
	yasm.RegisterSignal("lb_wait_ack", unistat.NewNumeric("lb_wait_ack", 1, aggr.Counter(), unistat.Sum))
}
