package metrics

import (
	"fmt"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/httputil/headers"
	"a.yandex-team.ru/library/go/yandex/unistat"
)

const (
	UnistatPort = 5321
)

var (
	Axxx = unistat.StructuredAggregation{AggregationType: unistat.Absolute, Group: unistat.Max, MetaGroup: unistat.Max, Rollup: unistat.Max}
	Annn = unistat.StructuredAggregation{AggregationType: unistat.Absolute, Group: unistat.Min, MetaGroup: unistat.Min, Rollup: unistat.Min}
	Avvv = unistat.StructuredAggregation{AggregationType: unistat.Absolute, Group: unistat.Average, MetaGroup: unistat.Average, Rollup: unistat.Average}
)

type Yasm struct {
	unistat.Registry
	server  *http.Server
	logger  *zap.Logger
	signals map[string]*unistat.Numeric
}

func NewYasm(logger *zap.Logger) *Yasm {
	registry := unistat.NewRegistry()
	mux := http.NewServeMux()
	mux.HandleFunc(
		"/unistat",
		func(w http.ResponseWriter, r *http.Request) {
			bytes, err := registry.MarshalJSON()
			if err != nil {
				http.Error(w, err.Error(), http.StatusInternalServerError)
				logger.Error("Cannot marshal metrics JSON", log.Error(err))
				return
			}

			w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
			if _, err := w.Write(bytes); err != nil {
				logger.Error("cannot write unistat response", log.Error(err))
			}
		})

	server := http.Server{
		Addr:    fmt.Sprintf(":%v", UnistatPort),
		Handler: mux,
	}

	go func() {
		logger.Fatal("Error listening to unistat server", log.Error(server.ListenAndServe()))
	}()

	return &Yasm{registry, &server, logger, make(map[string]*unistat.Numeric)}
}

func (y *Yasm) RegisterSignal(name string, signal *unistat.Numeric) {
	y.signals[name] = signal
	y.Register(signal)
}

func (y *Yasm) Update(name string, value float64) {
	if sig, ok := y.signals[name]; ok {
		sig.Update(value)
	}
}
