package storage

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/config"
	"a.yandex-team.ru/mail/logconsumers/actdb_consumer/metrics"
	"github.com/jackc/pgx/v4"
)

type IStorage interface {
	Connect(cfg *config.Config, yasm *metrics.Yasm, logger *zap.Logger) (*Storage, error)
	Run(sql string, yasm *metrics.Yasm) (bool, error)
}

type Storage struct {
	IStorage
	conn   *pgx.Conn
	logger *zap.Logger
}

func (s *Storage) Connect(cfg *config.Config, yasm *metrics.Yasm, logger *zap.Logger) (*Storage, error) {
	ctx := context.Background()
	if s.conn != nil {
		return s, nil
	}
	s.logger = logger

	logger.Infof("Storage connecting to %s", cfg.DB.Host)

	url := fmt.Sprintf("postgres://%s:%s@%s:%d/%s?application_name=actdb_consumer", cfg.DB.User, cfg.DB.Pass, cfg.DB.Host, cfg.DB.Port, cfg.DB.Name)
	conn, err := pgx.Connect(ctx, url)
	if err != nil {
		return s, err
	}

	if err := conn.Ping(ctx); err != nil {
		return s, err
	}

	s.conn = conn

	if ok, err := s.Run("SELECT 1", yasm); !ok {
		return s, err
	}

	return s, nil
}

func (s *Storage) Close() {
	ctx := context.Background()
	if err := s.conn.Close(ctx); err != nil {
		s.logger.Error("Error closing storage", log.Error(err))
	} else {
		s.logger.Info("Storage closed")
	}
	time.Sleep(250 * time.Millisecond)
}

func (s *Storage) Run(sql string, yasm *metrics.Yasm) (bool, error) {
	yasm.Update("db_queries", 1)
	ctx := context.Background()
	tx, err := s.conn.BeginTx(ctx, pgx.TxOptions{})
	if err != nil {
		s.logger.Error("Transaction begin failed", log.Error(err))
		yasm.Update("db_errors", 1)
		return false, err
	}

	_, err = tx.Exec(ctx, sql)
	if err != nil {
		s.logger.Error("Exec transaction failed", log.Error(err))

		// Incase we find any error in the query execution, rollback the transaction
		err = tx.Rollback(ctx)
		if err != nil {
			s.logger.Warn("Rollback transaction failed", log.Error(err))
		}

		yasm.Update("db_errors", 1)
		return false, err
	}

	err = tx.Commit(ctx)
	if err != nil {
		s.logger.Error("Failed commit transaction", log.Error(err))
		yasm.Update("db_errors", 1)
		return false, err
	}

	s.logger.Debugf("Run sql %s", sql)
	return true, nil
}
