#pragma once

#include <logdog/type_traits.h>
#include <logdog/variant.h>

#include <boost/hana/core/is_a.hpp>
#include <boost/hana/string.hpp>
#include <boost/hana/pair.hpp>
#include <boost/hana/drop_while.hpp>
#include <boost/hana/size.hpp>
#include <boost/hana/minus.hpp>
#include <boost/hana/not_equal.hpp>
#include <boost/hana/ext/std/tuple.hpp>
#include <boost/hana/ext/std/integral_constant.hpp>

#include <system_error>
#include <chrono>

namespace logdog {

using boost::optional;
namespace hana = boost::hana;

template <typename T>
struct is_optional_helper : std::false_type {};
template <typename T>
struct is_optional_helper<optional<T>> : std::true_type {};
template <typename T>
using is_optional = typename is_optional_helper<std::decay_t<T>>::type;
template <typename T>
constexpr auto is_optional_v = is_optional<T>::value;

template <typename T>
struct static_optional {
    T& v;
    using value_type = std::decay_t<T>;
    using is_none = std::is_same<value_type, none_t>;
    constexpr decltype(auto) operator * () const {return get();}
    constexpr decltype(auto) operator -> () const {return std::addressof(get());}
    constexpr operator bool () const {return !!(*this);}
    constexpr is_none operator !() const { return {};}
    constexpr decltype(auto) get() const { return v;}
};

template <typename Key, typename Value>
using attribute = hana::pair<Key, Value>;

template <typename Key, typename Value>
constexpr decltype(auto) key(const attribute<Key, Value>& attr) {
    return hana::first(attr);
}

template <typename Key, typename = hana::when<true>>
struct key_name_impl {
    static constexpr auto apply(const Key&) {
        return Key::name;
    }
};

template <typename Key>
constexpr decltype(auto) key_name(const Key& v) {
    static_assert(is_key_type_v<Key>, "Key is not a key type");
    static_assert(hana::is_a<hana::string_tag, decltype(key_name_impl<Key>::apply(v))>,
        "logdog::key_name_impl<Key>::apply() should return hana::string");
    return key_name_impl<Key>::apply(v);
}

constexpr decltype(auto) name(const none_t&) { return none;}

template <typename Key>
constexpr auto name(const Key& v) -> std::enable_if_t<is_key_type_v<Key>, const char*> {
    return hana::to<const char*>(key_name(v));
}

template <typename Key, typename Value>
constexpr decltype(auto) name(const attribute<Key, Value>& v) {
    return name(key(v));
}

namespace detail {

template <typename T>
struct unwrap_cref_impl {
    constexpr static const T& apply(const T& v) {return v;}
};

template <typename T>
struct unwrap_cref_impl<std::reference_wrapper<T>> : unwrap_cref_impl<T> {};

template <typename Value>
constexpr decltype(auto) unwrap_cref(const Value& v) {
    return unwrap_cref_impl<Value>::apply(v);
}

} // namespace impl

template <typename Key, typename Value>
constexpr decltype(auto) value(const attribute<Key, Value>& v) {
    return detail::unwrap_cref(hana::second(v));
}

constexpr none_t value(const none_t&) {return {};}

template <typename Key, typename Value>
inline decltype(auto) make_attribute(const Key& key, Value&& v) {
    static_assert(is_key_type_v<Key>, "Key must meet is_key_type requirements");
    return hana::make_pair(key, std::forward<Value>(v));
}

namespace impl {
template <typename T>
constexpr decltype(auto) as_optional(T& v) {
    if constexpr (!is_optional_v<T>) {
        return static_optional<T>{v};
    } else {
        return v;
    }
}

} // namespace impl

namespace detail {
template <typename Key>
struct key_is_not_equal {
    constexpr key_is_not_equal(const Key&) {};
    template <typename OtherKey, typename Value>
    constexpr auto operator() (const attribute<OtherKey, Value>& x) const ->
        decltype(logdog::key_name(logdog::key(x))!=logdog::key_name(std::declval<const Key&>()));
    template <typename OtherKey, typename Value>
    constexpr auto operator() (const optional<attribute<OtherKey, Value>>& x) const ->
        decltype((*this)(*x));
    constexpr std::true_type operator() (...) const;
};
} // namespace detail

template <typename Key, typename Sequence>
constexpr auto find_attribute(const Sequence& s, const Key& key) {
    const auto size = decltype(hana::size(s)){};
    const auto dropped = decltype(hana::size(hana::drop_while(s, detail::key_is_not_equal(key)))){};
    return size - dropped;
}

template <typename Key, typename Sequence>
constexpr auto has_attribute(const Sequence& s, const Key& key) {
    return std::bool_constant<decltype(find_attribute(s, key) != hana::size(s))::value>{};
}

template <typename Key, typename Sequence>
constexpr decltype(auto) get_attribute(const Sequence& s, const Key& key) {
    if constexpr (decltype(has_attribute(s, key))::value) {
        return impl::as_optional(hana::at(s, find_attribute(s, key)));
    } else {
        return impl::as_optional(none);
    }
}

template <typename Key, typename Sequence>
constexpr decltype(auto) get_value(const Sequence& s, const Key& key) {
    static_assert(!is_optional_v<decltype(get_attribute(s, key))>,
        "can not use get_value(Sequence, key) for optional attribute since dereference check"
        " is needed at run-time, use get_attribute() and perform the check instead");
    return value(get_attribute(s, key).get());
}

template <typename T, typename Name, typename = hana::when<true>>
struct attribute_definition {
    static constexpr Name name;
    using type = std::decay_t<T>;

    decltype(auto) operator = (const type& v) const {
        return make_attribute(*this, v);
    }

    template <typename Other>
    decltype(auto) operator = (const std::reference_wrapper<Other>& v) const {
        static_assert(std::is_base_of<type, std::decay_t<Other>>::value,
                "Other must be derived from or equal to the attribute_definition::type");
        return make_attribute(*this, std::cref(v.get()));
    }

    decltype(auto) operator = (type&& v) const {
        return make_attribute(*this, std::move(v));
    }

    decltype(auto) operator = (const none_t&) const {
        return make_optional(*this);
    }

    template <typename Sequence>
    constexpr decltype(auto) operator() (const Sequence& s) const {
        return get_value(s, *this);
    }
};

template <typename T, typename Name>
struct attribute_definition<T, Name, hana::when<is_variant<T>::value>> {
    static constexpr Name name;
    using type = std::decay_t<T>;

    decltype(auto) operator = (const type& v) const {
        return make_attribute(*this, v);
    }

    template <typename Other, typename = std::enable_if_t<std::is_same<type, std::decay_t<Other>>::value>>
    decltype(auto) operator = (const std::reference_wrapper<Other>& v) const {
        return make_attribute(*this, std::cref(v.get()));
    }

    decltype(auto) operator = (type&& v) const {
        return make_attribute(*this, std::move(v));
    }

    decltype(auto) operator = (const none_t&) const {
        return make_optional(*this);
    }

    template <typename Sequence>
    constexpr decltype(auto) operator() (const Sequence& s) const {
        return get_value(s, *this);
    }
};

template <typename T, typename Name>
struct attribute_definition<T, Name, hana::when<std::is_lvalue_reference<T>::value>> {
    static constexpr Name name;
    using type = std::remove_reference_t<T>;

    template <typename Other>
    decltype(auto) operator = (const std::reference_wrapper<Other>& v) const {
        static_assert(std::is_base_of<std::decay_t<type>, std::decay_t<Other>>::value,
                "Other must be derived from or equal to the attribute_definition::type");
        return make_attribute(*this, std::cref(v.get()));
    }

    decltype(auto) operator = (std::add_const_t<type>& v) const {
        return make_attribute(*this, std::cref(v));
    }

    decltype(auto) operator = (const none_t&) const {
        return make_optional(*this);
    }

    decltype(auto) operator = (type&& v) const = delete;

    template <typename Sequence>
    constexpr decltype(auto) operator() (const Sequence& s) const {
        return get_value(s, *this);
    }
};

template <typename ...Ts>
struct is_key_type<attribute_definition<Ts...>> : std::true_type {};

template <template<typename...> class C, typename Name>
struct template_attribute_definition {
    static constexpr Name name;
    template <typename ...Ts>
    using type = C<Ts...>;

    template <typename ...Ts>
    decltype(auto) operator = (const type<Ts...>& v) const {
        return make_attribute(*this, v);
    }

    template <typename ...Ts>
    decltype(auto) operator = (type<Ts...>&& v) const {
        return make_attribute(*this, std::move(v));
    }

    template <typename ...Ts>
    decltype(auto) operator = (const std::reference_wrapper<const type<Ts...>>& v) const {
        return make_attribute(*this, v);
    }

    template <typename ...Ts>
    decltype(auto) operator = (const std::reference_wrapper<type<Ts...>>& v) const {
        return make_attribute(*this, std::cref(v.get()));
    }

    template <typename Sequence>
    constexpr decltype(auto) operator() (const Sequence& s) const {
        return get_value(s, *this);
    }
};

template <template<typename ...> class C, typename Name>
struct is_key_type<template_attribute_definition<C, Name>> : std::true_type {};

template <typename T, typename Name>
constexpr attribute_definition<T, Name> define_attribute([[maybe_unused]] const Name& name) {
    static_assert(decltype(hana::is_a<hana::string_tag>(name))::value,
            "Attribute name type should be boost::hana::string");
    return {};
}
template <template<typename ...> class C, typename Name>
constexpr template_attribute_definition<C, Name> define_attribute([[maybe_unused]] const Name& name) {
    static_assert(decltype(hana::is_a<hana::string_tag>(name))::value,
            "Attribute name type should be boost::hana::string");
    return {};
}

template <typename Attr, typename Value = typename Attr::type&>
auto make_optional(const Attr& attr)
        -> optional<decltype(attr=std::declval<Value>())> {
    static_assert(is_key_type_v<Attr>,
        "Attr must meet is_key_type requirements");
    return boost::none;
}

template <typename Sequence, typename Key, typename Value>
void update_attribute(Sequence& s, const Key& key, Value&& v) {
    static_assert(decltype(has_attribute(s, key))::value, "attribute Key is not found in Sequence");
    hana::at(s, find_attribute(s, key)) = key=std::forward<Value>(v);
}

#define LOGDOG_DEFINE_ATTRIBUTE(Type, Name)\
    constexpr static auto Name = ::logdog::define_attribute<Type>(BOOST_HANA_STRING(#Name));

#ifdef LOGDOG_CONFIG_ENABLE_ATTRIBUTE_UDL

namespace detail {

template <typename Name>
struct attribute_definition_proxy {
    constexpr attribute_definition_proxy(const Name&) {}
    template <typename T>
    auto operator = (T&& v) const {
        return ::logdog::define_attribute<T>(Name{}) = std::forward<T>(v);
    }
    auto operator = (const char* v) const {
        return ::logdog::define_attribute<std::string_view>(Name{}) = v;
    }
};

} // namespace detail

template <typename Name>
struct key_name_impl<detail::attribute_definition_proxy<Name>> {
    static constexpr Name apply(const detail::attribute_definition_proxy<Name>&) {
        return {};
    }
};

template <typename Name>
struct is_key_type<detail::attribute_definition_proxy<Name>> : std::true_type {};

namespace literals {

template <class CharT, CharT ... c>
constexpr auto operator ""_a() {
    return logdog::detail::attribute_definition_proxy(hana::string<c ...>());
}

} // namespace literals

#endif

} // namespace logdog
