#pragma once

#include <boost/hana/at.hpp>
#include <boost/hana/tuple.hpp>
#include <boost/hana/size.hpp>
#include <boost/hana/equal.hpp>
#include <boost/hana/integral_constant.hpp>
#include <boost/hana/plus.hpp>
#include <boost/hana/minus.hpp>
#include <boost/hana/less.hpp>

namespace logdog::flatten {
namespace hana = boost::hana;
namespace detail {

template <typename Sequence, typename Idx = decltype(hana::size_c<0>), typename OwnerPtr = std::nullptr_t>
struct pointer {
    constexpr pointer() noexcept {}
    constexpr pointer(const Sequence&, Idx = Idx{}, OwnerPtr = nullptr) noexcept {}
    constexpr operator bool () const noexcept { return true;}
};

template <typename Sequence, typename Idx, typename Owner>
constexpr Idx idx(pointer<Sequence, Idx, Owner>) { return {};}

template <typename Sequence, typename Idx, typename OwnerPtr>
constexpr OwnerPtr owner(pointer<Sequence, Idx, OwnerPtr>) { return {};}

template <typename Sequence, typename Idx, typename Owner>
constexpr Sequence sequence_type(pointer<Sequence, Idx, Owner>);

template <typename Pointer>
constexpr auto value_type(Pointer p) -> std::decay_t<decltype(hana::at(sequence_type(p), idx(p)))>;

template <typename Pointer>
constexpr bool is_null = std::is_same_v<Pointer, std::nullptr_t>;

template <typename Sequence, typename Pointer>
constexpr decltype(auto) get(Sequence&& s, Pointer p) {
    static_assert(!is_null<Pointer>, "null pointer dereference");
    if constexpr (!is_null<decltype(owner(p))>) {
        return hana::at(get(std::forward<Sequence>(s), owner(p)), idx(p));
    } else {
        static_assert(std::is_same_v<std::decay_t<Sequence>, decltype(sequence_type(p))>,
            "Pointer should have same sequence_type as Sequence");
        return hana::at(std::forward<Sequence>(s), idx(p));
    }
}

template <typename Pointer>
constexpr auto advance_to_value(Pointer p) {
    if constexpr (!hana::Sequence<decltype(value_type(p))>::value) {
        return p;
    } else {
        return decltype(advance_to_value(pointer(value_type(p), hana::size_c<0>, p))){};
    }
}

template <typename Pointer>
constexpr auto next(Pointer p) {
    if constexpr (!is_null<Pointer>) {
        if constexpr (decltype(idx(p)+hana::size_c<1> < hana::size(sequence_type(p)))::value) {
            return decltype(advance_to_value(pointer(sequence_type(p), idx(p)+hana::size_c<1>, owner(p)))){};
        } else {
            return flatten::detail::next(owner(p));
        }
    } else {
        return p;
    }
}


template <typename Pointer, typename ...Ts>
constexpr auto make_pointers_sequence_type(Pointer p, Ts... vs) {
    if constexpr (!is_null<Pointer>) {
        return make_pointers_sequence_type(flatten::detail::next(p), vs..., p);
    } else {
        return hana::make_tuple(vs...);
    }
}

template <typename Sequence>
constexpr auto get_pointers(Sequence& s) ->
        decltype(make_pointers_sequence_type(advance_to_value(pointer(s)))) {
    return {};
}

} // namespace

template <typename Sequence>
constexpr auto copy(Sequence&& s) {
    using namespace detail;
    return hana::unpack(get_pointers(s), [&s](auto... p) {
        return hana::make_tuple(get(std::forward<Sequence>(s), p)...);
    });
}

template <typename Sequence>
constexpr auto view(const Sequence& s) {
    using namespace detail;
    return hana::unpack(get_pointers(s), [&s](auto... p) {
        return hana::tuple<decltype(get(s, p))&...>(get(s, p)...);
    });
}

} // namespace flatten
