#pragma once
#include <logdog/logger.h>
#include <logdog/level.h>
#include <logdog/format/tskv.h>

#include <mail/yreflection/include/yamail/data/serialization/yajl.h>
#include <mail/yreflection/include/yamail/data/reflection/reflection.h>

#include <boost/hana/for_each.hpp>
#include <boost/hana/map.hpp>
#include <boost/system/error_code.hpp>
#include <boost/fusion/include/adapt_struct.hpp>

#include <thread>

namespace logdog {
namespace json {

template <typename T>
struct get_representation_impl {
    static const T& apply(const T& v) noexcept { return v; }
};

struct error_code {
    std::string message;
    int value;
    std::string_view category;

    template <typename T>
    error_code(const T& ec)
            : message(ec.message()), value(ec.value()), category(ec.category().name()) {}
    error_code(const mail_errors::error_code& ec)
            : message(ec.what().empty() ? ec.message() : ec.base().message() + ": " + ec.what()),
              value(ec.value()), category(ec.category().name()) {}
};

struct exception {
    std::string what;
    std::string type;

    template <typename T>
    exception(const T& e)
            : what(e.what()), type(boost::core::demangle(typeid(e).name())) {}
};

template <>
struct get_representation_impl<boost::system::error_code> {
    static auto apply(const boost::system::error_code& v) noexcept { return error_code(v); }
};

template <>
struct get_representation_impl<std::error_code> {
    static auto apply(const std::error_code& v) noexcept { return error_code(v); }
};

template <>
struct get_representation_impl<mail_errors::error_code> {
    static auto apply(const mail_errors::error_code& v) noexcept { return error_code(v); }
};

template <>
struct get_representation_impl<std::exception> {
    static auto apply(const std::exception& v) noexcept { return exception(v); }
};

template <>
struct get_representation_impl<std::chrono::system_clock::time_point> {
    static auto apply(const std::chrono::system_clock::time_point& v) noexcept {
        return logdog::tskv::to_string(v);
    }
};

template <>
struct get_representation_impl<std::thread::id> {
    static auto apply(const std::thread::id& v) noexcept {
        std::stringstream ss;
        ss << v;
        return ss.str();
    }
};

template <typename Rep, typename Period>
struct get_representation_impl<std::chrono::duration<Rep, Period>> {
    static auto apply(const std::chrono::duration<Rep, Period>& v) noexcept { return v.count(); }
};

template <typename Name>
struct get_representation_impl<logdog::level_type<Name>> {
    static auto apply(const logdog::level_type<Name>& v) noexcept { return boost::hana::to<const char*>(v.name); }
};

template <typename T>
decltype(auto) get_representation(const T& v, std::enable_if_t<!is_variant<T>::value>* = nullptr) {
    return get_representation_impl<T>::apply(v);
}

template <typename T>
decltype(auto) get_representation(const T& v, std::enable_if_t<is_variant<T>::value>* = nullptr) {
    return visit([&](auto& v) {
        return get_representation(deref(v));
    }, v);
}

constexpr auto formatter = [] (const auto& args) {
    static_assert(boost::hana::Sequence<decltype(args)>::value, "args should model Hana.Sequence");
    auto m = boost::hana::transform(args, [](auto&& attr) {
        return boost::hana::make_pair(logdog::key(attr).name, get_representation(logdog::value(attr)));
    });
    return yamail::data::serialization::toJson(boost::hana::to<boost::hana::map_tag>(m)).str();
};

} // namespace json
} // namespace logdog

BOOST_FUSION_ADAPT_STRUCT(logdog::json::error_code,
    message,
    value,
    category
);

BOOST_FUSION_ADAPT_STRUCT(logdog::json::exception,
    what,
    type
)
