#pragma once

#include <logdog/logger.h>

#include <yplatform/tskv/tskv.h>

#include <boost/uuid/uuid_io.hpp>
#include <boost/core/demangle.hpp>
#include <boost/hana/core/is_a.hpp>
#include <boost/hana/string.hpp>
#include <boost/hana/for_each.hpp>

#include <functional>
#include <typeinfo>
#include <iomanip>

namespace logdog {
namespace tskv {

// ISO 8601 and useconds compatible format
// 1970-01-01T00:00:00.000000+0000
// No cheap tests due to global locale dependency, sorry :(
inline std::string to_string(const std::chrono::system_clock::time_point& v) {
    using namespace std::chrono;
    const auto sec = duration<double>(v.time_since_epoch() % minutes(1)).count();
    const auto tt = system_clock::to_time_t(v);
    std::tm tm;
    localtime_r(&tt, &tm);
    std::ostringstream s;
    char cstr[32];
    strftime(cstr, sizeof(cstr), "%FT%R:", &tm);
    s << cstr;
    s << std::fixed << std::setfill('0') << std::setw(9) << sec;
    strftime(cstr, sizeof(cstr), "%z", &tm);
    s << cstr;
    return s.str();
}

template <typename T, typename = hana::when<true>>
struct to_tskv_impl {
    template <typename Out>
    static void apply(Out& out, const char* key, const T& v) {
        out << ytskv::attr(key, v);
    }
};

template <typename Out, typename T>
inline void to_tskv(Out& out, const char* key, const T& v) {
    to_tskv_impl<std::decay_t<T>>::apply(out, key, v);
}

template <typename T>
struct to_tskv_impl<T, hana::when_valid<decltype(to_string(std::declval<const T&>()))>> {
    template <typename Out>
    static void apply(Out& out, const char* key, const T& v) {
        to_tskv(out, key, to_string(v));
    }
};

template <>
struct to_tskv_impl<std::chrono::steady_clock::duration> {
    template <typename Out>
    static void apply(Out& out, const char* key, const std::chrono::steady_clock::duration& v) {
        out << ytskv::attr(key, v.count());
    }
};

template <>
struct to_tskv_impl<std::vector<std::string>> {
    template <typename Out>
    static void apply(Out& out, const char* key, const std::vector<std::string>& v) {
        out << ytskv::attr(key, "[" + boost::join(v, ",") + "]");
    }
};

template <>
struct to_tskv_impl<std::list<std::string>> {
    template <typename Out>
    static void apply(Out& out, const char* key, const std::list<std::string>& v) {
        out << ytskv::attr(key, "[" + boost::join(v, ",") + "]");
    }
};

template <typename T>
struct error_code_to_tskv_impl {
    template <typename Out>
    static void apply(Out& out, const char* key, const T& ec) {
        const std::string name = key;
        out << ytskv::attr(name + ".category", ec.category().name())
            << ytskv::attr(name + ".value", ec.value())
            << ytskv::attr(name + ".message", ec.message());
    }
};

template <>
struct error_code_to_tskv_impl<mail_errors::error_code> {
    template <typename Out>
    static void apply(Out& out, const char* key, const mail_errors::error_code& ec) {
        const std::string name = key;
        out << ytskv::attr(name + ".category", ec.category().name())
            << ytskv::attr(name + ".value", ec.value())
            << ytskv::attr(name + ".message",
                           ec.what().empty() ? ec.message() : ec.base().message() + ": " + ec.what());
    }
};

template <>
struct to_tskv_impl<std::error_code> :
    error_code_to_tskv_impl<std::error_code> {};

template <>
struct to_tskv_impl<boost::system::error_code> :
    error_code_to_tskv_impl<boost::system::error_code> {};

template <>
struct to_tskv_impl<mail_errors::error_code> :
    error_code_to_tskv_impl<mail_errors::error_code> {};

template <>
struct to_tskv_impl<std::exception> {
    template <typename Out>
    static void apply(Out& out, const char* key, const std::exception& e) {
        const std::string name = key;
        out << ytskv::attr(name + ".type", boost::core::demangle(typeid(e).name()))
            << ytskv::attr(name + ".what", e.what());
    }
};

namespace impl {

template <typename T>
inline decltype(auto) has_value(const T& v, std::enable_if_t<
        is_nullable<T>::value>* = nullptr ) {
    return v;
}

template <typename T>
constexpr std::true_type has_value(const T&, std::enable_if_t<
        !is_nullable<T>::value>* = nullptr) {return {};}

} // namespace impl

template <typename Key, typename Value>
constexpr decltype(auto) has_value(const attribute<Key, Value>& v) {
    return impl::has_value(::logdog::value(v));
}

constexpr std::false_type has_value(const none_t&) {return {};}


template <typename Out>
struct attributes_visitor {
    Out& tskv;

    template <typename T>
    void operator () (const optional<T>& v) const {
        if (v) {
            (*this)(v.get());
        }
    }

    template <typename Key, typename Value>
    void operator () (const attribute<Key, Value>& v) const {
        if (has_value(v)) {
            using logdog::tskv::to_tskv;
            this->to_tskv(name(v), deref(value(v)));
        }
    }

    template <typename T>
    constexpr void operator () (T&&) const {
        static_assert(std::is_same<T, void>::value,
            "non-attribute item in logging sequence");
    }

    template <typename T>
    std::enable_if_t<!is_variant<T>::value> to_tskv(const char* key, const T& v) const {
        using logdog::tskv::to_tskv;
        to_tskv(tskv, key, v);
    }

    template <typename T>
    std::enable_if_t<is_variant<T>::value> to_tskv(const char* key, const T& v) const {
        visit([&](auto& v){
            using logdog::tskv::to_tskv;
            to_tskv(tskv, key, deref(v));
        }, v);
    }
};

template <typename Name>
struct formatter_type {
    static constexpr Name name;

    constexpr formatter_type(const Name&) {}

    template <typename Sequence>
    std::string operator() (const Sequence& args) const {
        static_assert(hana::Sequence<Sequence>::value, "args should model Hana.Sequence");
        std::ostringstream s;
        auto tskv = ytskv::utf_value(s);
        tskv << ytskv::attr("tskv_format", hana::to<const char*>(*this));
        hana::for_each(args, attributes_visitor<decltype(tskv)>{tskv});
        tskv << ytskv::endl;
        return s.str();
    }
};

template <typename Name>
constexpr auto make_formatter(const Name& name) {
    return formatter_type{name};
}

constexpr static auto formatter = make_formatter(BOOST_HANA_STRING("tskv-log"));

} // namespace tskv
} // namespace logdog

namespace boost::hana {

template <typename T>
struct to_impl<const char*, ::logdog::tskv::formatter_type<T>> {
    static constexpr const char* apply(const ::logdog::tskv::formatter_type<T>& v) {
        if constexpr (decltype(is_a<string_tag>(v.name))::value) {
            return to<const char*>(v.name);
        } else {
            return v.name.value;
        }

    }
};

} // namespace boost::hana
