#pragma once

#include <logdog/type_traits.h>

#include <utility>
#include <iostream>

namespace logdog {

template <typename T>
struct level_type;

template <typename Logger, typename = hana::when<true>>
struct applicable_impl {
    template <typename L, typename T>
    constexpr static decltype(auto) call(L&& logger, const level_type<T>& l) {
        return logger.applicable(l);
    }
};

template <typename Logger>
struct applicable_impl<std::reference_wrapper<Logger>> {
    template <typename L, typename T>
    constexpr static decltype(auto) call(L&& logger, const level_type<T>& l) {
        return logger.get().applicable(l);
    }
};

template <typename Logger, typename T>
constexpr decltype(auto) applicable(Logger&& logger, const level_type<T>& l) {
    return applicable_impl<std::decay_t<Logger>>::call(
        std::forward<Logger>(logger), l);
}

template <typename Logger>
struct applicable_impl<Logger, hana::when<is_nullable<std::decay_t<Logger>>::value>> {
    template <typename L, typename T>
    constexpr static bool call(L&& logger, const level_type<T>& l) {
        return logger && applicable(deref(logger), l);
    }
};

struct non_applicable {
    template <typename ...Ts>
    constexpr static std::false_type call(Ts&& ...) { return {};}
};

template <>
struct applicable_impl<none_t>  : non_applicable {};
template <>
struct applicable_impl<std::nullptr_t>  : non_applicable {};

template <typename Logger, typename = hana::when<true>>
struct write_impl {
    template <typename L, typename T, typename ...Ts>
    static decltype(auto) call(L&& logger, const level_type<T>& l, Ts&& ...args) {
        return logger.write(l, std::forward<Ts>(args)...);
    }
};

template <typename Logger>
struct write_impl<std::reference_wrapper<Logger>> {
    template <typename L, typename T, typename ...Ts>
    static decltype(auto) call(L&& logger, const level_type<T>& l, Ts&& ...args) {
        return logger.get().write(l, std::forward<Ts>(args)...);
    }
};

template <typename Logger>
struct write_impl<Logger, hana::when<is_nullable<std::decay_t<Logger>>::value>> {
    template <typename L, typename T, typename ...Ts>
    static decltype(auto) call(L&& logger, const level_type<T>& l, Ts&& ...args) {
        return deref(logger).write(l, std::forward<Ts>(args)...);
    }
};

struct non_write {
    template <typename ...Ts>
    constexpr static decltype(auto) call(Ts&& ...) { return none;}
};

template <>
struct write_impl<none_t>  : non_write {};
template <>
struct write_impl<std::nullptr_t>  : non_write {};

template <typename Formatter>
struct format_impl {
    template <typename F, typename Sequence>
    static decltype(auto) call(F&& formatter, Sequence&& sequence) {
        return formatter(std::forward<Sequence>(sequence));
    }
};

struct bypass_format {
    template <typename F, typename Sequence>
    constexpr static decltype(auto) call(F&& , Sequence&& sequence) {
        return std::forward<Sequence>(sequence);
    }
};

template <>
struct format_impl<none_t>  : bypass_format {};
template <>
struct format_impl<std::nullptr_t>  : bypass_format {};

template <typename Logger>
struct write_data_impl {
    template <typename L, typename T, typename Data>
    static decltype(auto) call(L&& logger, const level_type<T>& level,
            Data&& data) {
        return logger.write_data(level, std::forward<Data>(data));
    }
};

template <>
struct write_data_impl<none_t>  : non_write {};
template <>
struct write_data_impl<std::nullptr_t>  : non_write {};

template <typename Logger, typename T, typename ...Ts>
inline decltype(auto) write(Logger&& l, const level_type<T>& level, Ts&& ...args) {
    return write_impl<std::decay_t<Logger>>::call(
        std::forward<Logger>(l),
        level,
        std::forward<Ts>(args)...);
}

template <typename Formatter, typename Sequence>
inline decltype(auto) format(Formatter&& f, Sequence&& s) {
    return format_impl<std::decay_t<Formatter>>::call(
        std::forward<Formatter>(f),
        std::forward<Sequence>(s));
}

template <typename Logger, typename T, typename Data>
inline decltype(auto) write_data(Logger&& logger, const level_type<T>& level, Data&& data) {
    return write_data_impl<std::decay_t<Logger>>::call(
        std::forward<Logger>(logger), level, std::forward<Data>(data));
}

/**
 * Log level abstraction. Encapsulates current log level acceptance check
 * and write to log with self level value.
 */
template <typename Name>
struct level_type {
    static constexpr Name name;

    constexpr level_type(const Name&) {}

    template <typename Log>
    struct writer {
        Log& log_;
        level_type level_;
        template <typename ...Args>
        void operator()(Args&& ...args) const {
            write(log_, level_, std::forward<Args>(args)...);
        }
    };

    template <typename Log, typename What>
    void operator() (Log&& log, What&& f) const {
        if (applicable(log, *this)) {
            f(writer<Log>{log, *this});
        }
    }
};

template <typename T1, typename T2>
constexpr auto operator == (const level_type<T1>&, const level_type<T2>&) {
    return std::bool_constant<decltype(level_type<T1>::name == level_type<T2>::name)::value>{};
}

template <typename T1, typename T2>
constexpr auto operator != (const level_type<T1>&, const level_type<T2>&) {
    return std::bool_constant<decltype(level_type<T1>::name != level_type<T2>::name)::value>{};
}

#define LOGDOG_DEFINE_LEVEL(Name)\
    constexpr static auto Name = ::logdog::level_type(BOOST_HANA_STRING(#Name));

LOGDOG_DEFINE_LEVEL(warning)
LOGDOG_DEFINE_LEVEL(error)
LOGDOG_DEFINE_LEVEL(notice)
LOGDOG_DEFINE_LEVEL(debug)

static_assert (decltype(warning == warning)::value, "levels with same name should be equal");
static_assert (decltype(warning != error)::value, "levels with different names should not be equal");

template <typename T>
constexpr decltype(auto) to_string(const level_type<T>& l) {
    return hana::to<const char*>(l);
}

template <typename T>
inline std::ostream& operator << (std::ostream& out, const level_type<T>& l) {
    return out << to_string(l);
}

} // namespace logdog

namespace boost::hana {

template <typename T>
struct to_impl<string_tag, ::logdog::level_type<T>> {
    static constexpr auto apply(::logdog::level_type<T>) {
        return ::logdog::level_type<T>::name;
    }
};

template <typename T>
struct to_impl<const char*, ::logdog::level_type<T>> {
    static constexpr const char* apply(::logdog::level_type<T> v) {
        return to<const char*>(to<string_tag>(v));
    }
};

} // namespace boost::hana
