#pragma once

#include <logdog/attribute.h>
#include <logdog/level.h>
#include <logdog/type_traits.h>
#include <logdog/variant.h>
#include <logdog/detail/flatten.h>

#include <mail_errors/error_code.h>

#include <boost/uuid/uuid.hpp>

#include <boost/hana/tuple.hpp>
#include <boost/hana/unpack.hpp>

#include <chrono>
#include <thread>
#include <tuple>

namespace logdog {

namespace attr {
LOGDOG_DEFINE_ATTRIBUTE(std::exception&, exception)
using errors_variant = make_attr_variant<
                            std::error_code,
                            boost::system::error_code,
                            mail_errors::error_code>;
LOGDOG_DEFINE_ATTRIBUTE(errors_variant, error_code)
LOGDOG_DEFINE_ATTRIBUTE(std::string, message)
LOGDOG_DEFINE_ATTRIBUTE(level_type, level)
LOGDOG_DEFINE_ATTRIBUTE(std::time_t, unixtime)
LOGDOG_DEFINE_ATTRIBUTE(std::chrono::system_clock::time_point, timestamp)
LOGDOG_DEFINE_ATTRIBUTE(std::thread::id, thread)

LOGDOG_DEFINE_ATTRIBUTE(std::string, where_name)
LOGDOG_DEFINE_ATTRIBUTE(std::string, where_file)
LOGDOG_DEFINE_ATTRIBUTE(std::string, where_line)

} // namespace attr

using namespace attr;

namespace filters {

template <typename Next>
struct expand_system_error {
    using system_error = boost::system::system_error;

    template <typename Continuation, typename ...Ts>
    constexpr static decltype(auto) apply(Continuation&& cont, Ts&& ...args) {
        const auto tied = std::tie(args...);
        const auto attrs = flatten::view(tied);

        if (!has_attribute(attrs, attr::error_code)) {
            if (auto ex = get_attribute(attrs, attr::exception)) {
                if (auto err = get_system_error<system_error>(value(*ex))) {
                    return Next::apply(std::forward<Continuation>(cont),
                        attr::error_code=err->code(), std::forward<Ts>(args)...);
                } else if (auto err = get_system_error<std::system_error>(value(*ex))) {
                    return Next::apply(std::forward<Continuation>(cont),
                        attr::error_code=err->code(), std::forward<Ts>(args)...);
                }
            }
        }
        return Next::apply(std::forward<Continuation>(cont), std::forward<Ts>(args)...);
    }

    template <typename Error>
    constexpr static const Error* get_system_error(const Error& e) {
        return std::addressof(e);
    }
    template <typename Error, typename T>
    constexpr static const Error* get_system_error(const T& e) {
        return dynamic_cast<const Error*>(std::addressof(e));
    }
    template <typename Error>
    constexpr static const Error* get_system_error(const none_t&) {
        return nullptr;
    }
};

template <typename Next>
struct add_timestamp {
    template <typename Continuation, typename ...Ts>
    constexpr static decltype(auto) apply(Continuation&& cont, Ts&& ...args) {
        const auto attrs = flatten::view(std::tie(args...));
        static_assert(!has_attribute(attrs, attr::unixtime), "Attribute unixtime is added twice!");
        static_assert(!has_attribute(attrs, attr::timestamp), "Attribute timestamp is added twice!");
        using clock = std::chrono::system_clock;
        const auto t = clock::now();
        return Next::apply(std::forward<Continuation>(cont),
            attr::unixtime=clock::to_time_t(t), attr::timestamp=t,
            std::forward<Ts>(args)...);
    }
};

template <typename Next>
struct add_thread_id {
    template <typename Continuation, typename ...Ts>
    constexpr static decltype(auto) apply(Continuation&& cont, Ts&& ...args) {
        const auto attrs = flatten::view(std::tie(args...));
        static_assert(!has_attribute(attrs, attr::thread), "Attribute thread is added twice!");
        return Next::apply(std::forward<Continuation>(cont),
            attr::thread=std::this_thread::get_id(),
            std::forward<Ts>(args)...);
    }
};

template <typename>
struct terminator {
    template <typename Continuation, typename ...Ts>
    constexpr static decltype(auto) apply(Continuation&& cont, Ts&& ...args) {
        return cont(std::forward<Ts>(args)...);
    }
};

template <template<typename> class ...Tps>
using make_sequence = typename monadic_fold_left<Tps..., terminator>::type;

using default_sequence = make_sequence<expand_system_error, add_timestamp, add_thread_id>;

} // namespace filters

template <typename Format, typename Backend, typename Filters>
class log {
public:
    log(Format format, Backend backend)
    : format_(format), backend_(backend) {}

    template <typename T>
    decltype(auto) applicable(const level_type<T>& l) const {
        using ::logdog::applicable;
        return applicable(backend_, l);
    }

    template <typename T, typename ...Args>
    decltype(auto) write(const level_type<T>& l, Args&& ...args) const {
        return apply_filters([&](auto&& ...vs) {
            const auto data = std::make_tuple(std::forward<decltype(vs)>(vs)...);
            return write_data(deref(this->backend_), l,
                format(this->format_, flatten::view(data)));
        }, attr::level=l, std::forward<Args>(args)...);
    }
private:

    template <typename Continuation, typename ...Ts>
    constexpr static decltype(auto) apply_filters(Continuation c, Ts&&... vs) {
        return Filters::apply(std::move(c), std::forward<Ts>(vs)...);
    }

    Format format_;
    Backend backend_;
};

template <typename Format, typename Backend, typename Filters = filters::default_sequence>
inline auto make_log(Format format, Backend backend, Filters = Filters{}) {
    return log<Format, Backend, Filters>(std::move(format), std::move(backend));
}

template <typename Log, typename Context>
class context_binder {
public:
    using context_type = Context;
    using underlying_log_type = Log;

    context_binder(Log log, Context ctx)
    : log_(std::move(log)), ctx_(std::move(ctx)) {}

    context_type& get_context() noexcept { return ctx_;}
    const context_type& get_context() const noexcept { return ctx_;}

    underlying_log_type& get_underlying_log() noexcept { return log_;}
    const underlying_log_type& get_underlying_log() const noexcept { return log_;}

    template <typename T>
    decltype(auto) applicable(const level_type<T>& l) const {
        using ::logdog::applicable;
        return applicable(get_underlying_log(), l);
    }

    template <typename T, typename ...Args>
    decltype(auto) write(const level_type<T>& l, Args&& ...args) const {
        return hana::unpack(get_context(), [&](auto&... ctx) {
            using ::logdog::write;
            return write(get_underlying_log(), l, ctx..., std::forward<Args>(args)...);
        });
    }

private:
    underlying_log_type log_;
    context_type ctx_;
};

template <typename Log, typename ...Ts>
inline constexpr decltype(auto) bind(Log&& log, Ts&&... vs) {
    return context_binder{std::forward<Log>(log), hana::make_tuple(std::forward<Ts>(vs)...)};
}

template <typename T1, typename T2, typename ...Ts>
inline constexpr decltype(auto) bind(context_binder<T1, T2>&& log, Ts&&... vs) {
    return context_binder {
        std::move(log.get_underlying_log()),
        hana::unpack(std::move(log.get_context()), [&](auto&& ...ctx) {
            return hana::make_tuple(std::move(ctx)..., std::forward<Ts>(vs)...);
        })
    };
}

template <typename T1, typename T2, typename ...Ts>
inline constexpr decltype(auto) bind(const context_binder<T1, T2>& log, Ts&&... vs) {
    return context_binder {
        log.get_underlying_log(),
        hana::unpack(log.get_context(), [&](auto ...ctx) {
            return hana::make_tuple(std::move(ctx)..., std::forward<Ts>(vs)...);
        })
    };
}

} // namespace logdog

/**
 * This macro is to substitute the boilerplate of lambda only. This is not
 * a general interface, so if you need something more or something else
 * do not histate and use the general interface with lambda.
 */
#define LOGDOG_(Logger, Level, ...)\
    ::logdog::Level(Logger, [&](auto __lgw) mutable {__lgw(__VA_ARGS__);});

#define __LOGDOG_STRING_LINE_I_(x) #x
#define __LOGDOG_STRING_LINE(x) __LOGDOG_STRING_LINE_I_(x)
#define __LOGDOG_MAKE_WHERE ::std::make_tuple(\
    ::logdog::where_name=__FUNCTION__,\
    ::logdog::where_file=__FILE__,\
    ::logdog::where_line=__LOGDOG_STRING_LINE(__LINE__))

/**
 * This macro allows you to log the source coordinates there
 * the macro is called.
 */
#define LOGDOG_WHERE_(Logger, Level, ...) \
    ::logdog::Level(Logger, [&, __lgw_w_ = __LOGDOG_MAKE_WHERE](auto __lgw) mutable {\
        __lgw(__VA_ARGS__, __lgw_w_);});
