#pragma once

#include <logdog/hana.h>
#include <boost/optional.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/hana/core/when.hpp>
#include <type_traits>
#include <optional>
#include <memory>

namespace logdog {

struct none_t {};
constexpr none_t none;

template <typename T>
struct is_nullable : std::is_pointer<T> {};
template <typename T>
struct is_nullable<boost::optional<T>> : std::true_type {};
template <typename T>
struct is_nullable<std::optional<T>> : std::true_type {};
template <typename T>
struct is_nullable<std::shared_ptr<T>> : std::true_type {};
template <typename T>
struct is_nullable<std::unique_ptr<T>> : std::true_type {};
template <typename T>
struct is_nullable<boost::shared_ptr<T>> : std::true_type {};
template <typename T>
struct is_nullable<boost::scoped_ptr<T>> : std::true_type {};

template <typename T>
struct is_key_type : std::false_type {};

template <typename T>
constexpr bool is_key_type_v = is_key_type<T>::value;

template <typename T>
struct is_derefereceable : std::integral_constant<bool,
    is_nullable<T>::value &&
    !std::is_same<T, const char*>::value
> {};

namespace impl {

template <typename T, typename = hana::when<true>>
struct deref_impl {
    template <typename Arg>
    constexpr static auto& apply(Arg&& v) { return v; }
};

template <typename T>
struct deref_impl<std::reference_wrapper<T>> {
    template <typename Arg>
    constexpr static auto& apply(Arg&& v) { return v.get(); }
};

template <typename T>
struct deref_impl<T, hana::when<is_derefereceable<T>::value>> {
    template <typename Arg>
    constexpr static auto& apply(Arg&& v) { return *v; }
};

} // namespace impl

template <typename T>
constexpr auto& deref(T&& v) {
    return impl::deref_impl<std::decay_t<T>>::apply(std::forward<T>(v));
}

template <template<typename> class ...Tps>
struct monadic_fold_left;

template <template<typename> class Tp1, template<typename> class Tp2, template<typename> class ...Tps>
struct monadic_fold_left<Tp1, Tp2, Tps...> {
    using type = Tp1<typename monadic_fold_left<Tp2, Tps...>::type>;
};

template <template<typename> class Tp>
struct monadic_fold_left<Tp> {
    using type = Tp<void>;
};

} // namespace logdog
