#pragma once

#include <macs/types.h>
#include <macs/tab.h>
#include <optional>
#include <cstdint>
#include <variant>
#include <string>
#include <vector>
#include <ctime>


namespace macs {

enum class BackupState {
    in_progress,
    complete,
    inactive,
    error
};

enum class RestoreState {
    in_progress,
    complete,
    error
};

enum class RestoreMethod {
    full_hierarchy,
    restored_folder
};

enum class CreateBackupStatus {
    empty_folder_list = ENUM_ERROR_STARTING_VALUE,
    nothing_to_backup,
    too_many_messages,
    running_restore,
    unique_violation,
    backup_id_not_reserved
};

enum class RestoreBackupStatus {
    unique_violation = ENUM_ERROR_STARTING_VALUE,
    backup_is_missing,
    another_backup_is_running,
    wrong_state
};

enum class FillBackupStatus {
    wrong_state = ENUM_ERROR_STARTING_VALUE,
    nothing_to_backup,
    backup_is_missing
};

enum class DeactivateBackupStatus {
    restore_in_progress = ENUM_ERROR_STARTING_VALUE,
    missing_complete_backup
};

using BackupId = int32_t;

using BackupFidsMapping = std::map<macs::Fid, macs::Fid>;
using BackupFidToFidVec = std::vector<std::pair<int32_t, int32_t>>;

using BackupMidToMid = std::map<Mid, Mid>;
using BackupMidToMidVec = std::vector<std::pair<int64_t, int64_t>>;

struct Backup {
    BackupId backup_id;
    std::time_t created;
    std::time_t updated;
    int32_t message_count;
    BackupState state;
    std::optional<std::string> notice;
    int32_t version;
};

struct Restore {
    BackupId backup_id;
    std::time_t created;
    std::time_t updated;
    int32_t to_restore_count;
    int32_t restored_count;
    RestoreState state;
    RestoreMethod method;
    std::optional<std::string> notice;
    BackupFidsMapping fids_mapping;
};

struct BackupStatus {
    std::optional<Backup> primary;
    std::optional<Backup> additional;
    std::optional<Restore> restore;
};

struct BackupSettings {
    std::vector<Fid> fids;
    std::vector<Tab::Type> tabs;
};

struct BackupMessage {
    Mid mid;
    std::string st_id;
    Fid fid;
    std::optional<Tab::Type> tab;
    std::time_t receivedDate;
    std::vector<std::string> attributes;
};

using BackupBox = std::vector<BackupMessage>;

struct BackupFolder {
    macs::Fid fid;
    std::string name;
    std::optional<macs::Fid> parentFid;
    bool isBackuped = false;
};

using BackupFolders = std::vector<BackupFolder>;

}
