#pragma once

#include <macs/hooks.h>
#include <macs/io.h>
#include <macs/types.h>
#include <macs/check_arguments.h>

namespace macs {

class BackupsRepository {
public:
    virtual ~BackupsRepository() = default;

    template <typename Handler = io::sync_context>
    auto getStatus(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnBackupStatus> init(handler);
        asyncStatus(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto createBackup(macs::BackupId id, int32_t maxMessages, bool useTabs, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncCreateBackup(id, maxMessages, useTabs, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto fillBackup(macs::BackupId id, bool useTabs, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncFillBackup(id, useTabs, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto restore(macs::BackupId id, std::time_t now, macs::RestoreMethod method,
                 macs::BackupFidsMapping mapping, Handler handler = io::use_sync) const {
        ASSERT_BIGINT_MAP_ITEMS(mapping);
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncRestore(id, now, std::move(method), std::move(mapping), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto backupId(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnBackupId> init(handler);
        asyncId(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getSettings(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnBackupSettings> init(handler);
        asyncSettings(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto changeSettings(const BackupSettings& settings, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnBackupSettings> init(handler);
        asyncChangeSettings(settings, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto folders(macs::BackupId id, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnBackupFolders> init(handler);
        asyncFolders(id, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto selectMessages(macs::BackupId id, uint64_t rowCount, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnBackupBox> init(handler);
        asyncSelectMessages(id, rowCount, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto selectMessagesInFolder(macs::BackupId id, uint64_t rowCount, macs::Fid fid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnBackupBox> init(handler);
        asyncSelectMessagesInFolder(id, rowCount, fid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deleteMessages(macs::BackupId id, macs::MidVec mids, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncDeleteMessages(id, std::move(mids), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateMessages(macs::BackupId id, macs::BackupMidToMid midToMid, Handler handler = io::use_sync) const {
        ASSERT_BIGINT_MAP_ITEMS(midToMid);
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncUpdateMessages(id, std::move(midToMid), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto failRunningRestore(std::string notice, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncFailRunningRestore(std::move(notice), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto completeRunningRestore(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncCompleteRunningRestore(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto failRunningBackup(std::string notice, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncFailRunningBackup(std::move(notice), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deactivateBackup(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncDeactivateBackup(init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncStatus(OnBackupStatus) const = 0;
    virtual void asyncCreateBackup(macs::BackupId, int32_t, bool, OnExecute) const = 0;
    virtual void asyncFillBackup(macs::BackupId, bool, OnExecute) const = 0;
    virtual void asyncRestore(macs::BackupId, std::time_t, macs::RestoreMethod, macs::BackupFidsMapping, OnExecute) const = 0;
    virtual void asyncFolders(macs::BackupId, OnBackupFolders) const = 0;
    virtual void asyncId(OnBackupId) const = 0;
    virtual void asyncSettings(OnBackupSettings) const = 0;
    virtual void asyncChangeSettings(const BackupSettings&, OnBackupSettings) const = 0;
    virtual void asyncSelectMessages(macs::BackupId, uint64_t, OnBackupBox) const = 0;
    virtual void asyncSelectMessagesInFolder(macs::BackupId, uint64_t, macs::Fid hiddenFid, OnBackupBox) const = 0;
    virtual void asyncDeleteMessages(macs::BackupId, macs::MidVec, OnExecute) const = 0;
    virtual void asyncUpdateMessages(macs::BackupId, macs::BackupMidToMid, OnExecute) const = 0;
    virtual void asyncCompleteRunningRestore(OnExecute) const = 0;
    virtual void asyncFailRunningRestore(std::string, OnExecute) const = 0;
    virtual void asyncFailRunningBackup(std::string, OnExecute) const = 0;
    virtual void asyncDeactivateBackup(OnExecute) const = 0;
};

using BackupsRepositoryPtr = std::shared_ptr<BackupsRepository>;

}
