#pragma once

#include <algorithm>
#include <boost/lexical_cast/try_lexical_convert.hpp>
#include <macs/errors.h>

namespace macs {

template <typename Arg>
inline void checkEmptyArg(const Arg& arg, const char* name, const char* func)  {
    if (arg.empty()) {
        std::ostringstream s;
        s << "empty " << name << " is forbidden in " << func;
        throw ParamsException(s.str());
    }
}

template <typename ArgList>
inline void checkEmptyItemInArgList(const ArgList& list, const char * name, const char * func)  {
    for (const auto& item: list) {
        if (item.empty()) {
            std::ostringstream s;
            s << "empty elements in " << name << " list are forbidden in " << func;
            throw ParamsException(s.str());
        }
    }
}

template <typename NumericType>
inline bool isNumeric(const std::string& str) {
    NumericType res;
    return boost::conversion::try_lexical_convert(str, res);
}

template <typename NumericType>
inline void checkNonNumericArg(const std::string& arg, const char * name, const char * func)  {
    if (!isNumeric<NumericType>(arg)) {
        std::ostringstream s;
        s << "bad value '" << arg << "' for " << name << " param in " << func;
        throw ParamsException(s.str());
    }
}

template <typename NumericType, typename ArgList>
inline void checkNonNumericItemInArgList(const ArgList& list, const char * name, const char * func)  {
    auto badParam = std::find_if_not(list.cbegin(), list.cend(), isNumeric<NumericType>);
    if (badParam != list.cend()) {
        std::ostringstream s;
        s << "bad element '" << *badParam << "' in " << name << " list in " << func;
        throw ParamsException(s.str());
    }
}

template <typename NumericType, typename ArgMap>
inline void checkNonNumericItemInArgMap(const ArgMap& map, const char * name, const char * func)  {
    for (const auto& pair: map) {
        if (!isNumeric<NumericType>(pair.first)) {
            std::ostringstream s;
            s << "bad key element '" << pair.first << "," << pair.second << "' in " << name << " map in " << func;
            throw ParamsException(s.str());
        }

        if (!isNumeric<NumericType>(pair.second)) {
            std::ostringstream s;
            s << "bad value element '" << pair.first << "," << pair.second << "' in " << name << " map in " << func;
            throw ParamsException(s.str());
        }

    }
}

#define ASSERT_NOT_EMPTY_ARG(arg) checkEmptyArg(arg, #arg, __PRETTY_FUNCTION__)
#define ASSERT_NO_EMPTY_ITEMS(arg) checkEmptyItemInArgList(arg, #arg, __PRETTY_FUNCTION__)
#define ASSERT_BIGINT_ARG(arg) checkNonNumericArg<int64_t>(arg, #arg, __PRETTY_FUNCTION__)
#define ASSERT_INTEGER_ARG(arg) checkNonNumericArg<int32_t>(arg, #arg, __PRETTY_FUNCTION__)
#define ASSERT_BIGINT_ITEMS(arg) checkNonNumericItemInArgList<int64_t>(arg, #arg, __PRETTY_FUNCTION__)
#define ASSERT_BIGINT_MAP_ITEMS(arg) checkNonNumericItemInArgMap<int64_t>(arg, #arg, __PRETTY_FUNCTION__)
#define ASSERT_INTEGER_ITEMS(arg) checkNonNumericItemInArgList<int32_t>(arg, #arg, __PRETTY_FUNCTION__)

} // namespace macs
