#ifndef MACS_FOLDER_POSITION_H
#define MACS_FOLDER_POSITION_H

#include <boost/iterator/filter_iterator.hpp>
#include <macs/folder_set.h>

namespace macs {

const size_t FOLDER_POSITION_MULTIPLIER = 100;

class GetFolderPosition {
public:
    GetFolderPosition(const FolderSet& folderSet) : folderSet(folderSet) {}

    size_t operator() (const std::string& parentId) const {
        const size_t last = getLastPosition(parentId);
        return last == 0 ? 0 : last + FOLDER_POSITION_MULTIPLIER;
    }

private:
    const FolderSet& folderSet;

    struct ParentIs {
        const std::string * fid;
        ParentIs (const std::string& fid) : fid(&fid) {}
        bool operator() (const FolderSet::value_type& arg) const {
            return arg.second.parentId() == *fid;
        }
    };

    typedef boost::filter_iterator<ParentIs, FolderSet::const_iterator> Iter;

    static size_t getPosition(const FolderSet::value_type & arg) {
        return arg.second.position();
    }

    static bool less(const FolderSet::value_type& arg1, const FolderSet::value_type& arg2) {
          return getPosition(arg1) < getPosition(arg2);
    }

    Iter findFolderWithHighestPosition(const std::string& parentId) const {
        Iter first(parentId, folderSet.begin(), folderSet.end());
        Iter last(parentId, folderSet.end(), folderSet.end());
        return std::max_element(first, last, less);
    }

    size_t getLastPosition(const std::string& parentId) const {
        Iter i = findFolderWithHighestPosition(parentId);
        if (i.base() != folderSet.end() && getPosition(*i)) {
            return getPosition(*i);
        }
        return 0;
    }
};

} //namespace macs

#endif // MACS_FOLDER_POSITION_H

