#pragma once

#include <macs/data/folder_position.h>
#include <macs/folders_repository.h>
#include <macs/errors.h>
#include <boost/bind.hpp>
#include <butil/StrUtils/utfutils.h>

namespace macs {
namespace detail {

class SortOptionsImpl : public std::enable_shared_from_this<SortOptionsImpl> {
public:
    SortOptionsImpl(const FoldersRepository& repo) : repo_(repo) {}
private:
    static size_t getPosition( const Folder & folder ) {
        return folder.position();
    }

    static size_t getPosition( const FolderSet::value_type & arg ) {
        return getPosition( arg.second );
    }

    static bool compareFoldersAlpha(const Folder& f1, const Folder& f2) {
        std::string name1(f1.name());
        std::string name2(f2.name());
        utf::to_lower(name1);
        utf::to_lower(name2);
        return utf::to_wstring(name1) < utf::to_wstring(name2);
    }

    static bool compareFoldersPosition(const Folder& f1, const Folder& f2) {
        return getPosition(f1) < getPosition(f2);
    }

    typedef std::map<std::string, std::list<Folder> > Levels;

    Levels getLevelsMap() {
        FolderSet folderSet = getAllFolders();
        Levels levels;
        for (FolderSet::iterator i = folderSet.begin(); i != folderSet.end(); i++) {
            levels[i->second.parentId()].push_back(getFolderByFid(i->second.fid()));
        }
        return levels;
    }

    void setPosition(Folder& folder, size_t position) {
        FolderFactory factory(std::move(folder));
        factory.position(position);
        folder = factory.product();
    }

    void setPositionsForLevel(Levels::mapped_type& level) {
        size_t position = FOLDER_POSITION_MULTIPLIER;
        for (Levels::mapped_type::iterator i = level.begin(); i != level.end(); i++, position += FOLDER_POSITION_MULTIPLIER) {
            setPosition(*i, position);
            updateFolder(*i);
        }
    }

    template<typename F>
    void setAllPositions(F comparator, const std::string& parent) {
        Levels levels = getLevelsMap();
        Levels::mapped_type level = levels[parent];
        level.sort(comparator);
        setPositionsForLevel(level);
    }

    void firstTimePositionsPlacement(const std::string& parent) {
        setAllPositions(&compareFoldersAlpha, parent);
    }

    void rearrangePositions(const std::string& parent) {
        setAllPositions(&compareFoldersPosition, parent);
    }

    struct positionDefined {
        bool operator()(const Levels::mapped_type::value_type& arg) const {
            return getPosition( arg ) != 0;
        }
    };

    bool needInitialPositionsPlacement(const std::string& parent) {
        Levels levels = getLevelsMap();
        Levels::mapped_type level = levels[parent];
        return std::find_if( level.begin(), level.end(), positionDefined()) == level.end();
    }

    static bool folderIs(const std::string& fid, const Folder& folder) {
        return fid == folder.fid();
    }

    size_t getNextPosition(const std::string& parent, const std::string& prevFid) {
        Levels levels = getLevelsMap();
        Levels::mapped_type level = levels[parent];
        level.sort(&compareFoldersPosition);
        Levels::mapped_type::iterator iter = find_if(level.begin(), level.end(), boost::bind(&folderIs, prevFid, _1));
        if (iter == level.end()) {
            return getPosition(level.front());
        } else if (iter->fid() == level.back().fid()) {
            return getPosition(level.back()) + FOLDER_POSITION_MULTIPLIER;
        } else {
            return getPosition(*++iter);
        }
    }

    void getNeighboursPositions(const std::string& parent, const std::string& prevFid, size_t& prevPosition, size_t& nextPosition) {
        auto updatePrevPosition = [&]() mutable {
            const auto zeroPosition = prevFid == Folder::noParent || getFolderByFid(prevFid).parentId() != parent;
            prevPosition = zeroPosition ? 0 : getPosition(getFolderByFid(prevFid));
        };

        updatePrevPosition();
        nextPosition = getNextPosition(parent, prevFid);

        if (nextPosition - prevPosition <= 1) {
            rearrangePositions(parent);
            updatePrevPosition();
            nextPosition = prevPosition + FOLDER_POSITION_MULTIPLIER;
        }
    }
public:
    void move(const std::string& folderFid, const std::string& prevFid, OnCountReceive hook) {
        repo_.getAllFolders([self = shared_from_this(),
                             folderFid, prevFid,
                             hook = std::move(hook)](const auto ec, auto folders) {
            if (ec) {
                hook(ec);
            } else {
                try {
                    self->folders_ = std::move(folders);
                    Folder folder = self->getFolderByFid(folderFid);
                    if (self->needInitialPositionsPlacement(folder.parentId())) {
                        self->firstTimePositionsPlacement(folder.parentId());
                    }

                    if (folderFid != prevFid) {
                        size_t prevPosition, nextPosition;
                        self->getNeighboursPositions(folder.parentId(), prevFid, prevPosition, nextPosition);
                        const size_t folderPosition = (prevPosition + nextPosition) / 2;

                        self->setPosition(folder, folderPosition);
                        self->updateFolder(folder);
                    }

                    self->apply(getPosition(folder), std::move(hook));
                } catch (const boost::system::system_error& e) {
                    hook(e.code());
                }
            }
        });
    }

private:
    FolderSet & getAllFolders() { return folders_; }

    const Folder& getFolderByFid(const std::string & fid) {
        return modified_.count(fid) ? modified_.at(fid) : folders_.at(fid);
    }
    void updateFolder(const Folder & f) {
        modified_.erase(f.fid());
        modified_.insert({f.fid(), f});
    }
    void apply(const size_t position, OnCountReceive hook) {
        apply(position, std::move(hook), modified_.begin());
    }
    void apply(const size_t position, OnCountReceive hook, const FoldersMap::iterator iter) {
        if (iter == modified_.end()) {
            hook(position);
        } else {
            repo_.updateFolder(iter->second, [self = shared_from_this(),
                                              hook = std::move(hook),
                                              position, iter](const auto ec, auto) {
                if (ec) {
                    hook(ec);
                } else {
                    self->apply(position, std::move(hook), std::next(iter));
                }
            });
        }
    }
    const FoldersRepository& repo_;
    FolderSet folders_;
    FoldersMap modified_;

}; //class SortOptions

} // namespace detail

class SortOptions {
public:
    SortOptions(const FoldersRepository& repo) : repo_(repo) {}

    template <typename Handler = io_result::sync_context>
    auto move(const std::string& folderFid, const std::string& prevFid,
              Handler handler = io_result::use_sync) {
        io::detail::init_async_result<Handler, OnCountReceive> init(handler);
        auto impl = std::make_shared<detail::SortOptionsImpl>(repo_);
        impl->move(folderFid, prevFid, init.handler);
        return init.result.get();
    }

private:
    const FoldersRepository& repo_;
};

} //namespace macs

