#pragma once

#include <boost/bind.hpp>
#include <boost/range/algorithm/find_if.hpp>
#include <macs/detail/objects_dict.h>

namespace macs {

enum SymbolicFolderNames {
    SYM_ZOMBIE_FOLDER = -1, //!< artificial label for seen (!new) messages

    SYM_FOLDER_NONE = 0, //!< non-localized folder name

    // folders
    SYM_INBOX = 1, //!< folder is INBOX
    SYM_SENT = 2, //!< folder is SENT
    SYM_TRASH = 3, //!< folder is TRASH
    SYM_SPAM = 4, //!< folder is SPAM
    SYM_DRAFTS = 5, //!< forder is DRAFTS
    SYM_OUTBOX = 6, //!< folder is OUTBOX
    SYM_ARCHIVE = 7, //!< folder is ARCHIVE
    SYM_TEMPLATE = 8, //!< folder is TEMPLATE
    SYM_DISCOUNT = 9, //!< folder for discount messages
    SYM_UNSUBSCRIBE = 10, //!< folder for filltered messages (unsubscribe)
    SYM_PENDING = 11, //!< folder for pending mailing lists
    SYM_HIDDEN_TRASH = 12, //!< folder for hidden trash
    SYM_RESTORED = 13, //!< folder for restored messages
    SYM_REPLY_LATER = 14, //!< folder for "reply later" messages
};

enum SymbolicLabelNames {
    SYM_LABEL_NONE = 0, //!< non-localized label name

    // system labels
    SYM_ATTACHED_LBL = 10, //!< label for messages with attachments
    SYM_SPAM_LBL = 11, //!< label for spam messages
    SYM_ANSWERED_LBL = 12, //!< imap label \Answered
    SYM_RECENT_LBL = 13, //!< imap label \Recent
    SYM_DRAFT_LBL = 14, //!< imap label \Draft
    SYM_DELETED_LBL = 15, //!< imap label \Deleted
    SYM_FORWARDED_LBL = 16, //!< imap label $Forwarded
    SYM_IMPORTANT_LBL = 17, //!< label for important (imap: \flagged) messages
    SYM_FOR_ME_LBL = 18, //!< letter to me only
    SYM_NO_BODY_LBL = 19, //!< message body is absent in mulca
    //empty space
    SYM_NO_ANSWER_LBL = 21, //!< no answer in time
    SYM_HAS_USER_LABELS_LBL = 22, //!< message has user labels
    SYM_SEEN_LBL = 23, //!< artificial label for seen (!new) messages
    SYM_PINNED_LBL = 24, //!< artificial label for pinned messages
    SYM_POSTMASTER_LBL = 25, //!< represents postmaster flag
    SYM_MULCA_SHARED_LBL = 26, //!< represents mulca-shared flag
    SYM_SMS_LBL = 27, //!< represents sms flag
    SYM_IMAP_LBL = 28, //!< represents IMAP flag
    SYM_APPEND_LBL = 29, //!< represents append flag
    SYM_COPY_LBL = 30, //!< represents copy flag
    SYM_REMIND_NO_ANSWER_LBL = 31, //!< artificial label for messages waiting answer
    SYM_NOTIFY_NO_ANSWER_LBL = 32, //!< artificial label for reminder messages about no answer
    SYM_REMIND_MESSAGE_LBL = 33, //!< artificial label for messages waiting reminder
    SYM_NOTIFY_MESSAGE_LBL = 34, //!< artificial label for reminder messages
    SYM_MUTE_LBL = 35, //!< requires 'seen_label' for every new message in this thread
    SYM_DELAYED_MESSAGE_LBL = 36, //!< artificial label for delayed messages
    SYM_HAMON_LBL = 37, //!< artificial label for priority inbox (important messages aka hamon)
    SYM_SYNCED_LBL = 38, //!< message was synced by code.sync_message
    SYM_ENCRYPTED_LBL = 39,
    SYM_UNDO_MESSAGE_LBL = 40,
    SYM_MENTION_LBL = 41,
    SYM_MENTION_UNVISITED_LBL = 42,
    SYM_SENDING_FAILED_LBL = 43,
    SYM_FROM_FAVORITE_USER_LBL = 44,
    SYM_REPLY_LATER_STARTED = 45,
    SYM_REPLY_LATER_FINISHED = 46,
};

template <typename T>
class MultiIndexDict : public detail::ObjectsDict<T> {
public:
    typedef typename detail::ObjectsDict<T> Base;
    typedef typename Base::Dict Dict;
    typedef typename Dict::const_iterator Iterator;

    int code() const {
        return code_;
    }
    const std::string title() const {
        return title_;
    }

    static const T & getByTitle(const std::string & title) {
        Iterator found = find_if( Base::dict(), boost::bind( &T::title, _1 ) == title );
        return found == Base::dict().end() ? T::defValue() : **found;
    }
    static const T & getByCode(const int code) {
        Iterator found = find_if( Base::dict(), boost::bind( &T::code, _1 ) == code );
        return found == Base::dict().end() ? T::defValue() : **found;
    }

    bool operator==(const T & val) const {
        return this == &val || this->code_ == val.code_;
    }
    bool operator!=(const T & val) const {
        return this != &val && this->code_ != val.code_;
    }
    bool operator<(const T & val) const {
        return this->code_ < val.code_;
    }

protected:
    MultiIndexDict(const int code, const std::string & title) : code_(code), title_(title) {}

private:
    int code_ = 0;
    std::string title_;
};

}

