#ifndef INCLUDE_MACS_DETAIL_CACHE_H_
#define INCLUDE_MACS_DETAIL_CACHE_H_

#include <memory>
#include <mutex>

namespace macs {
namespace detail {

/**
 * Simple thread-safe cache for non-modified data.
 */
template <typename T>
class Cache {
public:
    using Value = const T;
    using Ptr = std::shared_ptr<Value>;

    Ptr get() {
        std::lock_guard<std::mutex> l(m);
        return c;
    }

    Ptr set(T && v) {
        auto newval = std::make_shared<Value>(std::move(v));
        auto retval = newval;

        //Here we minimize the lock time. We do only pointers swap under the lock.
        //Old value destruction will be done out of lock if it is needed.
        std::unique_lock<std::mutex> l(m);
        std::swap(newval, c);
        l.unlock();

        return retval;
    }

    void reset() {
        Ptr tmp;
        std::unique_lock<std::mutex> l(m);
        std::swap(c, tmp);
        l.unlock();
    }
private:
    Ptr c;
    std::mutex m;
};

} // namespace detail
} // namespace macs

#endif /* INCLUDE_MACS_DETAIL_CACHE_H_ */
