#pragma once

/**
 * @file   envelope.h
 * @brief  Envelope - envelope headers
 *         (macs envelope struct wrapper)
 * @author Sergey Nikishin
 * © Yandex LLC.
 */

#include <string>
#include <map>
#include <set>
#include <vector>
#include <ctime>
#include <macs/types.h>
#include <macs/label.h>
#include <macs/tab.h>
#include <butil/email/email.h>

namespace macs {

struct AttachmentDescriptor {
    Hid m_hid;
    std::string m_contentType;
    std::string m_fileName;
    std::size_t m_size = 0;

    AttachmentDescriptor() = default;

    AttachmentDescriptor ( const Hid& hid, const std::string & contentType,
            const std::string& fileName, std::size_t size)
            : m_hid (hid),  m_contentType(contentType),
              m_fileName(fileName), m_size(size) {
    }
};

using Attachments = std::vector<AttachmentDescriptor>;
using Emails = std::vector<Email>;
using EmailsOpt = std::optional<Emails>;

struct EnvelopeData {
    Mid mid;                            //!< message ID in the metabase
    Fid fid;                            //!< ID of the folder that contains the message
    std::optional<Tab::Type> tab;       //!< Type of tab that contains the message
    Tid threadId;                       //!< ID of the thread that contains the message
    Revision revision;                  //!< mail revision
    std::time_t date = 0;               //!< the date when the message was sent
    std::time_t receiveDate = 0;        //!< date when the message was received
    std::string from;                   //!< 'From' header
    EmailsOpt parsedFrom;               //!< Parsed 'From' header
    std::string sender;                 //!< 'Sender' header
    EmailsOpt parsedSender;             //!< Parsed 'Sender' header
    std::string replyTo;                //!< 'Reply-To' header
    EmailsOpt parsedReplyTo;            //!< Parsed 'Reply-To' header
    std::string subject;                //!< 'Subject' header
    std::string cc;                     //!< 'CC' header
    EmailsOpt parsedCc;                 //!< Parsed 'CC' header
    std::string bcc;                    //!< 'BCC' header
    EmailsOpt parsedBcc;                //!< Parsed 'BCC' header
    std::string to;                     //!< 'To' header
    EmailsOpt parsedTo;                 //!< Parsed 'To' header
    std::string uidl;                   //!< uidl for pop3 server
    std::string imapId;                 //!< id for imap server
    std::size_t size = 0;               //!< message size in bytes
    std::string stid;                   //!< message ID in mulca
    std::string firstline;              //!< first line of the message
    std::vector<Lid> labels;            //!< list of IDs of labels on the message
    std::set<int> types;                //!< SO message types
    std::string inReplyTo;              //!< 'In-Reply-To' header
    std::string references;             //!< 'References' header
    RfcMessageId rfcId;                 //!< 'Message-ID' header
    Attachments attachments;            //!< descriptors of message attachments
    std::int32_t threadCount = 0;       //!< thread size in messages
    std::size_t attachmentsCount = 0;   //!< specified attachments count
    std::size_t attachmentsFullSize = 0; //!< specified attachments full size in bytes
    std::size_t newCount = 0;           //!< new messages count in thread
    std::string extraData;              //!< extra information

    static const EnvelopeData default_;
};

template <typename Impl>
class EnvelopeDataInterface {
public:
#define GETTER(NAME) auto& NAME () const { return data().NAME; }
    GETTER(mid)
    GETTER(fid)
    GETTER(tab)
    GETTER(threadId)
    GETTER(revision)
    GETTER(date)
    GETTER(receiveDate)
    GETTER(from)
    GETTER(parsedFrom)
    GETTER(sender)
    GETTER(parsedSender)
    GETTER(replyTo)
    GETTER(parsedReplyTo)
    GETTER(subject)
    GETTER(cc)
    GETTER(parsedCc)
    GETTER(bcc)
    GETTER(parsedBcc)
    GETTER(to)
    GETTER(parsedTo)
    GETTER(uidl)
    GETTER(imapId)
    GETTER(stid)
    GETTER(firstline)
    GETTER(inReplyTo)
    GETTER(references)
    GETTER(rfcId)
    GETTER(size)
    GETTER(threadCount)
    GETTER(extraData)
    GETTER(newCount)
    GETTER(attachments)
    GETTER(labels)
    GETTER(types)
#undef GETTER

    std::size_t attachmentsCount() const {
        return attachments().empty() ? data().attachmentsCount : attachments().size();
    }

    std::size_t attachmentsFullSize() const {
        return attachments().empty() ? data().attachmentsFullSize : calculateAttachmentsFullSize();
    }

    /// check if marked with label
    bool hasLabel(const Lid& lid) const {
        return std::find(labels().begin(), labels().end(), lid) != labels().end();
    }

    /// check if marked with label
    bool hasLabel(const Label& label) const {
        return hasLabel( label.lid() );
    }

private:
    // We do not want to allow to create an object of the class since it can not
    // provide functionality by itself. So there are two possible solutions:
    // 1 - make abstract class, 2 - make private constructors.
    // First solution leads to the virtual functions overhead and virtual d-tor.
    // Second solution allow us to make the class non-instantiable without any
    // virtual mechanism overhead.
    friend Impl;
    EnvelopeDataInterface() = default;
    EnvelopeDataInterface(const EnvelopeDataInterface&) = default;
    EnvelopeDataInterface(EnvelopeDataInterface&&) = default;
    EnvelopeDataInterface& operator = (const EnvelopeDataInterface&) = default;
    EnvelopeDataInterface& operator = (EnvelopeDataInterface&&) = default;

    std::size_t calculateAttachmentsFullSize() const {
        std::size_t size(0);
        for (const auto& att : attachments()) {
            size += att.m_size;
        }
        return size;
    }

    const EnvelopeData& data() const { return static_cast<const Impl&>(*this).Impl::data(); }
};


/** @class Envelope
 *  @brief envelope for letter (message)
 *         holds headers information
 *
 * contains copy of macs::data::message_list::entry,
 * for all modify operations use EnvelopesCollection objects
 */
class Envelope : public EnvelopeDataInterface<Envelope> {
public:
    /**
     * Message type constants. For the most recent info look at
     *   http://wiki.yandex-team.ru/BitovyeFlagiISMIXED
     */
    enum Type {
        Type_none = 0,
        Type_delivery = 1,
        Type_registration = 2,
        Type_social = 3,
        Type_people = 4,
        Type_eticket = 5,
        Type_eshop = 6,
        Type_notification = 7,
        Type_bounce = 8,
        Type_official = 9,
        Type_script = 10,
        Type_dating = 11,
        Type_greeting = 12,
        Type_news  = 13,
        Type_s_grouponsite = 14,
        Type_s_datingsite  = 15,
        Type_s_aviaeticket = 16,
        Type_s_bank = 17,
        Type_s_social = 18,
        Type_s_travel = 19,
        Type_s_zdticket  = 20,
        Type_s_realty = 21,
        Type_personalnews = 22,
        Type_s_eshop = 23,
        Type_s_company = 24,
        Type_s_job = 25,
        Type_s_game = 26,
        Type_schema = 27,
        Type_cancel = 28,
        Type_s_tech = 29,
        Type_s_media = 30,
        Type_s_advert = 31,
        Type_s_provider = 32,
        Type_s_forum = 33,
        Type_s_mobile = 34,
        Type_hotel = 35,
        Type_yamoney = 36,
    };

    enum Status {
        Status_read,
        Status_unread,
        Status_replied,
        Status_forwarded
    };
    Envelope() = default;
    Envelope(std::shared_ptr<const EnvelopeData> data) : data_(std::move(data)) {}
    Envelope(const Envelope&) = default;
    Envelope(Envelope&&) = default;
    Envelope& operator=(const Envelope&) = default;
    Envelope& operator=(Envelope&&) = default;
    /// represent envelope as string
    std::string toString() const;
private:
    friend class EnvelopeDataInterface<Envelope>;
    friend class EnvelopeFactory;
    const EnvelopeData& data() const {
        return data_ ? *data_ : EnvelopeData::default_;
    }
    std::shared_ptr<const EnvelopeData> data_;
};

std::string getJournalStatusString(Envelope::Status status);

class EnvelopeDeleted : public Envelope {
public:
    EnvelopeDeleted() : Envelope() {}
    EnvelopeDeleted(std::shared_ptr<const EnvelopeData> data) : Envelope(data) {}
};

std::string getJournalStatusString(EnvelopeDeleted::Status status);

} // namespace macs
