#pragma once

#include <ctime>
#include <macs/envelope.h>

namespace macs {

class EnvelopeFactory : private EnvelopeDataInterface<EnvelopeFactory>
{
    friend class EnvelopeDataInterface<EnvelopeFactory>;
public:
#define MODIFIER(NAME) \
    EnvelopeFactory& NAME (decltype(EnvelopeData::NAME) v) { \
        data().NAME = std::move(v); \
        return *this; \
    }
    MODIFIER( mid )
    MODIFIER( fid )
    MODIFIER( tab )
    MODIFIER( threadId )
    MODIFIER( revision )
    MODIFIER( date )
    MODIFIER( receiveDate )
    MODIFIER( from )
    MODIFIER( parsedFrom )
    MODIFIER( sender )
    MODIFIER( parsedSender )
    MODIFIER( to )
    MODIFIER( parsedTo )
    MODIFIER( replyTo )
    MODIFIER( parsedReplyTo )
    MODIFIER( subject )
    MODIFIER( cc )
    MODIFIER( parsedCc )
    MODIFIER( bcc )
    MODIFIER( parsedBcc )
    MODIFIER( uidl )
    MODIFIER( imapId )
    MODIFIER( size )
    MODIFIER( threadCount )
    MODIFIER( extraData )
    MODIFIER( stid )
    MODIFIER( firstline )
    MODIFIER( inReplyTo )
    MODIFIER( references )
    MODIFIER( rfcId )
    MODIFIER( newCount )
    MODIFIER( attachmentsCount )
    MODIFIER( attachmentsFullSize )
#undef MODIFIER

    EnvelopeFactory & addLabelID(Lid lid) {
        data().labels.push_back(std::move(lid));
        return *this;
    }

    EnvelopeFactory& addType(int type) {
        data().types.insert(type);
        return *this;
    }

    EnvelopeFactory& addLabelIDs(std::vector<Lid> lids) {
        data().labels.insert(data().labels.end(),
                std::make_move_iterator(lids.begin()),
                std::make_move_iterator(lids.end()));
        return *this;
    }
    EnvelopeFactory& addAttachment(AttachmentDescriptor attachment) {
        data().attachments.push_back(std::move(attachment));
        return *this;
    }

    EnvelopeFactory& removeLabel(const Lid& lid) {
        auto& labels = data().labels;
        auto it = std::remove(labels.begin(), labels.end(), lid);
        labels.erase(it, labels.end());
        return *this;
    }

    EnvelopeFactory& clearLabels() {
        data().labels.clear();
        return *this;
    }

    const EnvelopeDataInterface<EnvelopeFactory>& product() const {
        return *this;
    }

    EnvelopeFactory& reset() {
        data_.reset();
        return *this;
    }

    Envelope release() {
        removeDublicatesInLabels();
        auto retval = std::move(data_);
        reset();
        return Envelope{std::move(retval)};
    }

    EnvelopeDeleted releaseDeleted() {
        removeDublicatesInLabels();
        auto retval = std::move(data_);
        reset();
        return EnvelopeDeleted{std::move(retval)};
    }

    EnvelopeFactory() = default;

    EnvelopeFactory(const Envelope& src)
    : data_(std::make_shared<EnvelopeData>(src.data())) {
    }

    EnvelopeFactory(const EnvelopeFactory& ) = delete;
    EnvelopeFactory(EnvelopeFactory&&) = default;
    EnvelopeFactory& operator=(const EnvelopeFactory&) = delete;
    EnvelopeFactory& operator=(EnvelopeFactory&&) = default;
private:
    void removeDublicatesInLabels() {
        auto& labels = data().labels;
        std::sort ( labels.begin(), labels.end() );
        auto it = std::unique ( labels.begin(), labels.end() );
        labels.erase ( it, labels.end() );
    }

    EnvelopeData& data() {
        if(!data_) {
            data_ = std::make_shared<EnvelopeData>();
        }
        return *data_;
    }
    const EnvelopeData& data() const {
        return data_ ? *data_ : EnvelopeData::default_;
    }
    std::shared_ptr<EnvelopeData> data_;
};

}
