#pragma once

#include <list>
#include <boost/variant.hpp>
#include <boost/range/adaptors.hpp>
#include <macs/label.h>
#include <macs/envelope.h>
#include <macs/envelope_factory.h>
#include <macs/hooks.h>
#include <macs/queries/envelopes_sorting.h>
#include <macs/queries/envelopes_query_builder.h>
#include <macs/user_journal.h>
#include <macs/threads_meta.h>
#include <macs/check_arguments.h>
#include <macs/io.h>

namespace macs {

/// Envelopes access and manipulation class
class EnvelopesRepository : public std::enable_shared_from_this<EnvelopesRepository> {
public:
    EnvelopesRepository() = default;
    EnvelopesRepository(UserJournalPtr journal) : journal(journal) {}

    virtual ~EnvelopesRepository() = default;

    EnvelopesQueryBuilder query() {
        return EnvelopesQueryBuilder(*this);
    }

    /// Get one envelope by id, returns error if envelope doesn't exist
    template <typename Handler = io::sync_context>
    auto getById(const Mid & id, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<Envelope>> init(handler);
        getByIdInternal(id, init.handler);
        return init.result.get();
    }

    /// Get all envelopes by set of ids
    template <typename Handler>
    auto getByIds(const MidList& ids, Handler handler) const {
        ASSERT_NOT_EMPTY_ARG(ids);
        ASSERT_BIGINT_ITEMS(ids);
        io::detail::init_async_result<Handler, OnEnvelopeReceive> init(handler);
        syncEntriesByIds(ids, init.handler);
        return init.result.get();
    }

    /// label envelopes, mids and labels can not be empty
    template <typename Handler = io::sync_context>
    auto markEnvelopes(const std::list<Mid>& mids,
                       const std::list<Label>& labels,
                       Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        markEnvelopesInternal(mids, labels, init.handler);
        return init.result.get();
    }

    /// label envelopes in threads
    template <typename Handler = io::sync_context>
    auto markEnvelopesByThreads(const std::list<Tid>& tids,
                       const std::list<Label>& labels,
                       Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        markEnvelopesByThreadsInternal(tids, labels, init.handler);
        return init.result.get();
    }

    /// unlabel envelopes
    template <typename Handler = io::sync_context>
    auto unmarkEnvelopes(const std::list<Mid>& mids,
                         const std::list<Label>& labels,
                         Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        unmarkEnvelopesInternal(mids, labels, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto changeEnvelopesLabels(const std::list<Mid>& mids,
                               const std::list<Label>& labelsToAdd,
                               const std::list<Label>& labelsToDel,
                               Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        changeEnvelopesLabelsInternal(mids, labelsToAdd, labelsToDel, init.handler);
        return init.result.get();
    }

    /// unlabel envelopes, throws on storage error
    template <typename Handler = io::sync_context>
    auto unmarkEnvelopesByThreads(const std::list<Tid>& tids,
                         const std::list<Label>& labels,
                         Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        unmarkEnvelopesByThreadsInternal(tids, labels, init.handler);
        return init.result.get();
    }

    /// save new envelope in repository
    using ThreadMetaProvider = boost::function<ThreadMeta(const Envelope & e)>;

    enum class NotificationMode { on, off };
    enum class StoreType { box, deleted };
    struct SaveOptions {
        SaveOptions() = default;
        SaveOptions(bool ignoreDuplicates, NotificationMode notificationMode, StoreType storeType)
            : ignoreDuplicates(ignoreDuplicates),
            notificationMode(notificationMode),
            storeType(storeType) {}

        bool ignoreDuplicates{false};
        NotificationMode notificationMode{NotificationMode::on};
        StoreType storeType{StoreType::box};
    };

    template <typename Handler = io::sync_context>
    auto save(Envelope envelope, MimeParts mime, ThreadMetaProvider p,
               SaveOptions saveOptions, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateEnvelope> init(handler);
        saveInternal( std::move(envelope), std::move(mime), p, std::move(saveOptions), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto update( const Envelope & old, Envelope newer, MimeParts mime, ThreadMetaProvider p,
                 bool ignoreDuplicates, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateEnvelope> init(handler);
        updateInternal( old, std::move(newer), std::move(mime), p, ignoreDuplicates, init.handler);
        return init.result.get();
    }

    /// remove envelopes from repository
    template <typename Handler = io::sync_context>
    auto remove(const std::list<Mid>& mids, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        removeInternal(mids, false, init.handler);
        return init.result.get();
    }

    /// remove envelope from repository
    template <typename Handler = io::sync_context>
    auto remove(const Mid& mid, Handler handler = io::use_sync) const {
        return remove(std::list<Mid>{mid}, handler);
    }

    /// force remove envelopes from repository (guaranteed to deleted_box)
    template <typename Handler = io::sync_context>
    auto forceRemove(const std::list<Mid>& mids, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        removeInternal(mids, true, init.handler);
        return init.result.get();
    }

    /// force remove envelope from repository (guaranteed to deleted_box)
    template <typename Handler = io::sync_context>
    auto forceRemove(const Mid& mid, Handler handler = io::use_sync) const {
        return forceRemove(std::list<Mid>{mid}, handler);
    }

    /// get messages by thread
    void listInThread(const EnvelopesSorting & order,
                          const Tid& thread,
                          const boost::variant<std::size_t, string> & message,
                          const std::size_t count,
                          OnEnvelopeReceive handler) const;

    void listInThreadWithLabel(const Tid& threadId,
                               const macs::Label& label,
                               size_t from,
                               size_t count,
                               OnEnvelopeReceive handler) const;

    void listInThreadWithoutLabel(const Tid& threadId,
                                  const macs::Label& label,
                                  size_t from,
                                  size_t count,
                                  OnEnvelopeReceive handler) const;

    /// get messages for search
    void listFilterSearch(
                           bool unread,
                           bool atta,
                           const std::list<Fid>& fids,
                           const std::list<Lid>& lids,
                           const std::list<Mid>& mids,
                           const std::string& folderSet, // WTF???
                           const EnvelopesSorting & order,
                           OnEnvelopeReceive handler
                           ) const;

    /// get messages with no answer
    void listWithNoAnswer(const RfcMessageId& msgId, const Lid& includeLid,
                              OnEnvelopeReceive handler) const;

    /// get replies messages for given
    void listInReplyTo(const RfcMessageId& msgId, OnEnvelopeReceive handler) const;

    void listInReplyToByMid(const Mid& mid, OnEnvelopeReceive handler) const;

    template <typename Handler = io::sync_context>
    auto getFirstEnvelopeDate(const Fid& id, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUnixTime> init(handler);
        syncGetFirstEnvelopeDate(id, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getFirstEnvelopeDateOptional(const Fid& id, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUnixTimeOptional> init(handler);
        syncGetFirstEnvelopeDateOptional(id, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getFirstEnvelopeDateInTab(const Tab::Type& tab, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUnixTimeOptional> init(handler);
        syncGetFirstEnvelopeDateInTab(tab, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateMailAttributes(const std::string& mid,
                           const std::vector<std::string>& attributes,
                           Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        syncUpdateMailAttributes(mid, attributes, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getEnvelopesCount(const Fid& folderId,
                           std::time_t dateFrom,
                           std::time_t dateTo,
                           Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCountReceive> init(handler);
        syncGetEnvelopesCount(folderId, dateFrom, dateTo, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto resetFreshCounter(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        resetFreshCounterInternal(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto moveMessages(const Fid& dst, const std::list<Mid>& mids,
                      Handler handler = io::use_sync) const {
        return moveMessages(dst, std::nullopt, mids, handler);
    }

    template <typename Handler = io::sync_context>
    auto moveMessages(const Fid& dstFid,
                      const std::optional<Tab::Type>& dstTab,
                      const std::list<Mid>& mids,
                      Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        moveMessagesInternal(dstFid, dstTab, mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto moveMessage(const Fid& dst, const Mid& mid,
                     Handler handler = io::use_sync) const {
        return moveMessages(dst, {mid}, handler);
    }

    template <typename Handler = io::sync_context>
    auto copyMessages(const Fid& dst, const std::vector<Mid>& mids,
                      Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCopy> init(handler);
        copyMessagesInternal(dst, mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateStatus(const std::list<Mid>& mids, Envelope::Status status,
            Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        updateStatusInternal(mids, status, init.handler);
        return init.result.get();
    }

    void getMidsByFolder(const Fid& fid, OnMidsReceive handler) const {
        syncGetMidsByFolder(fid, handler);
    }
    void getMidsRangeByFolder(const Fid& fid, const std::optional<Mid>& fromMid, size_t count, OnMidsReceive handler) const {
        syncGetMidsRangeByFolder(fid, fromMid, count, handler);
    }
    void getMidsByFolderWithoutStatus(const Fid& fid,
                                      Envelope::Status excludeStatus,
                                      OnMidsReceive handler) const {
        syncGetMidsByFolderWithoutStatus(fid, excludeStatus, handler);
    }

    void getMidsRangeByFolderWithoutStatus(const Fid& fid,
                                           Envelope::Status excludeStatus,
                                           const std::optional<Mid>& fromMid,
                                           std::size_t count,
                                           OnMidsReceive handler) const {
        syncGetMidsRangeByFolderWithoutStatus(fid, excludeStatus, fromMid, count, handler);
    }

    void getMidsByTab(const Tab::Type& tab, OnMidsReceive handler) const {
        syncGetMidsByTab(tab, handler);
    }
    void getMidsByTabWithoutStatus(const Tab::Type& tab,
                                   Envelope::Status excludeStatus,
                                   OnMidsReceive handler) const {
        syncGetMidsByTabWithoutStatus(tab, excludeStatus, handler);
    }

    void getMidsByConditions(const EnvelopesQueryMidsByConditions& conds,
                                        OnMidsReceive handler) const  {
        syncGetMidsByConditions(conds, std::move(handler));
    }
    void getMidsByConditionsWithoutStatus(const EnvelopesQueryMidsByConditions& conds,
                                          Envelope::Status excludeStatus,
                                          OnMidsReceive handler) const {
        syncGetMidsByConditionsWithoutStatus(conds, excludeStatus, std::move(handler));
    }
    void getMidsByLabel(const Label& label, OnMidsReceive handler) const {
        syncGetMidsByLabel(label, handler);
    }

    template <typename Handler = io::sync_context>
    auto getMidsWithoutStatus(const MidList& mids,
                              Envelope::Status excludeStatus,
                              Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ITEMS(mids);
        io::detail::init_async_result<Handler, OnMidsReceive> init{handler};
        syncGetMidsWithoutStatus(mids, excludeStatus, init.handler);
        return init.result.get();
    }

    void getMidsByTidsAndLids(const Tids& tids, const Lids& lids, OnMidsReceive handler) const {
        ASSERT_BIGINT_ITEMS(tids);
        ASSERT_INTEGER_ITEMS(lids);
        syncGetMidsByTidsAndLids(tids, lids, std::move(handler));
    }

    template <typename Handler = io::sync_context>
    auto getAttachments(const Mid& mid, Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ARG(mid);
        io::detail::init_async_result<Handler, OnAttachmentsReceive> init{handler};
        syncGetAttachments(mid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMessageStIds(const MidList& mids, Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ITEMS(mids);
        io::detail::init_async_result<Handler, OnMidsWithStidsReceive> init{handler};
        syncGetMessageStIds(mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deleteFromStorage(const Stid& stid, Handler handler = io::use_sync) const{
        io::detail::init_async_result<Handler, OnExecute> init{handler};
        syncDeleteFromStorage(stid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getWindatMessageStId(const Mid& mid, const Hid& hid, Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ARG(mid);
        io::detail::init_async_result<Handler, OnStIdReceive> init{handler};
        syncGetWindatMessageStId(mid, hid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMimes(const Mids& mids, Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ITEMS(mids);
        io::detail::init_async_result<Handler, OnMidsWithMimes> init{handler};
        syncGetMimes(mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMimesWithAttaches(const Mids& mids, Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ITEMS(mids);
        io::detail::init_async_result<Handler, OnMidsWithMimesAndAttaches> init{handler};
        syncGetMimesWithAttaches(mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMimesWithDeleted(const Mids& mids, Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ITEMS(mids);
        io::detail::init_async_result<Handler, OnMidsWithMimes> init{handler};
        syncGetMimesWithDeleted(mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMimesWithAttachesWithDeleted(const Mids& mids, Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ITEMS(mids);
        io::detail::init_async_result<Handler, OnMidsWithMimesAndAttaches> init{handler};
        syncGetMimesWithAttachesWithDeleted(mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getWindatMimes(const Mids& mids, Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ITEMS(mids);
        io::detail::init_async_result<Handler, OnMidsWithMimes> init{handler};
        syncGetWindatMimes(mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto checkDuplicates(const Envelope & entry, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, CheckDuplicates> init{handler};
        syncCheckDuplicates(entry, init.handler);
        return init.result.get();
    }

    void threadsByTids(const TidVector& tids,
                           OnEnvelopeReceive handler) const;
    void threadsWithLabel(const Lid& label,
                              size_t from,
                              size_t count,
                              OnEnvelopeReceive handler) const;
    void threadsInFolderWithoutLabel(const Fid& folder,
                                         const Lid& label,
                                         size_t from,
                                         size_t count,
                                         OnEnvelopeReceive handler) const;
    void messagesInFolderWithoutLabel(const Fid& folder,
                                          const Lid& label,
                                          size_t from,
                                          size_t count,
                                          const EnvelopesSorting& order,
                                          OnEnvelopeReceive handler) const;

    template <typename Handler = io::sync_context>
    auto getMessageId(const Mid& mid, Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ARG(mid);
        io::detail::init_async_result<Handler, OnRfcMessageId> init{handler};
        syncGetMessageId(mid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getFreshCounter(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCountReceive> init{handler};
        syncGetFreshCounter(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getAttachesCounters(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnAttachesCounters> init{handler};
        syncGetAttachesCounters(init.handler);
        return init.result.get();
    }

    void envelopesQueryInMailbox(const Mid& fromMid,
                                 size_t from,
                                 size_t count,
                                 bool groups,
                                 const EnvelopesSorting& order,
                                 const std::list<Lid>& labels,
                                 const std::list<Lid>& excludeLabels,
                                 OnEnvelopeReceive handler) const;

    void envelopesQueryInFolder(const EnvelopesQueryInFolder::Params& params,
                                OnEnvelopeReceive handler) const;

    template <typename Handler = io::sync_context>
    auto getByMessageId(const Fid& fid, const RfcMessageId& messageId,
                        Handler handler = io::use_sync) const {
        ASSERT_INTEGER_ARG(fid);
        io::detail::init_async_result<Handler, OnMidsAndImapIds> init{handler};
        syncGetByMessageId(fid, messageId, init.handler);
        return init.result.get();

    }

    template <typename Handler = io::sync_context>
    auto getByMessageIdWithExcluded(const RfcMessageId& messageId,
                                    const FidVec& excludedFids,
                                    Handler handler = io::use_sync) const {
        ASSERT_INTEGER_ITEMS(excludedFids);
        io::detail::init_async_result<Handler, OnMidsReceive> init{handler};
        syncGetByMessageId(messageId, excludedFids, init.handler);
        return init.result.get();
    }

    void envelopesInFolderWithMimes(size_t from, size_t count, const EnvelopesSorting& order,
                                 const Fid& fid, OnEnvelopeWithMimeReceive handler) const;

    void envelopesByMidsWithMimes(const MidList& mids, OnEnvelopeWithMimeReceive handler) const;

    void getMidsByThreadAndWithSameHeaders(const Tid& threadId, OnMidsReceive handler) const {
        syncMidsByThreadAndWithSameHeaders(threadId, std::move(handler));
    }

    void getMidsByHdrDateAndMessageIdPairs(const EnvelopesQueryMidsByHdrPairs& hdrPairs,
                                           OnMidsReceive handler) const {
        syncMidsByHdrDateAndMessageIdPairs(hdrPairs, std::move(handler));
    }

    void envelopesInFolderWithMimesByChunks(size_t minImapId, size_t maxImapId, size_t chunkSize,
                                            const Fid& fid, OnEnvelopeWithMimeChunkReceive handler) const;

    void getDeletedMessages(size_t from, size_t count,
                            OnEnvelopeReceive handler) const;

    void getDeletedMessagesInInterval(size_t from, size_t count,
                                      const std::pair<std::time_t, std::time_t>& timeInterval,
                                      OnEnvelopeReceive handler) const;

    template <typename Handler = io::sync_context>
    auto getNewCountInFolderWithLabels(const Folder& folder,
                                       const std::vector<Label>& labels,
                                       size_t limit,
                                       Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCountReceive> init{handler};
        syncGetNewCountInFolderWithLabels(
                folder.fid(), lids(labels), limit, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getNewCountInFolderWithoutLabels(const Folder& folder,
                                          const std::vector<Label>& labels,
                                          size_t limit,
                                          Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCountReceive> init{handler};
        syncGetNewCountInFolderWithoutLabels(
                folder.fid(), lids(labels), limit, init.handler);
        return init.result.get();
    }

    void envelopesQueryInTab(const EnvelopesQueryInTab::Params& params,
                             OnEnvelopeReceive handler) const;

    template <typename Handler = io::sync_context>
    auto countNewMessagesWithLids(const std::list<Lid>& lids,
                                  size_t limit,
                                  Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCountReceive> init{handler};
        syncCountNewMessagesWithLids(lids, limit, init.handler);
        return init.result.get();
    }

protected:
    virtual void syncEnvelopesQueryInMailbox(const Mid& fromMid,
                                             size_t from,
                                             size_t count,
                                             bool groups,
                                             const EnvelopesSorting& order,
                                             const std::list<Lid>& labels,
                                             const std::list<Lid>& excludeLabels,
                                             OnEnvelopeReceive handler) const = 0;

    virtual void syncEnvelopesQueryInFolder(const EnvelopesQueryInFolder::Params& params,
                                            OnEnvelopeReceive handler) const = 0;

    /// get messages by thread
    virtual uint64_t syncListInThread(
                                      const EnvelopesSorting & order,
                                      const Tid & thread,
                                      const boost::variant<std::size_t, string> & message,
                                      const std::size_t count,
                                      OnEnvelopeReceive handler
                                      ) const = 0;

    virtual void syncListInThreadWithLabel(const Tid& threadId,
                                           const Label& label,
                                           size_t from,
                                           size_t count,
                                           OnEnvelopeReceive handler) const = 0;

    virtual void syncListInThreadWithoutLabel(const Tid& threadId,
                                              const Label& label,
                                              size_t from,
                                              size_t count,
                                              OnEnvelopeReceive handler) const = 0;

    /// get messages for search
    virtual uint64_t syncListFilterSearch(
                                       bool unread,
                                       bool atta,
                                       const std::list<Fid>& fids,
                                       const std::list<Lid>& lids,
                                       const std::list<Mid>& mids,
                                       const std::string& folderSet,
                                       const EnvelopesSorting & order,
                                       OnEnvelopeReceive handler
                                       ) const = 0;

    /// get messages with no answer
    virtual uint64_t syncListWithNoAnswer(const RfcMessageId& msgId,
                                          const Lid& includeLid,
                                          OnEnvelopeReceive handler) const = 0;

    /// get replies messages for given
    virtual uint64_t syncListInReplyTo(const RfcMessageId& msgId,
                                       OnEnvelopeReceive handler) const = 0;

    virtual uint64_t syncListInReplyToByMid(const Mid& mid,
                                            OnEnvelopeReceive handler) const = 0;

    virtual void syncGetFirstEnvelopeDate(const Fid&, OnUnixTime) const = 0;
    virtual void syncGetFirstEnvelopeDateOptional(const Fid&, OnUnixTimeOptional) const = 0;
    virtual void syncGetFirstEnvelopeDateInTab(const Tab::Type&, OnUnixTimeOptional) const = 0;

    virtual void syncUpdateMailAttributes(const std::string& mid,
                                          const std::vector<std::string> attributes,
                                          OnExecute hook) const = 0;

    virtual void syncGetEnvelopesCount(const Fid& folderId,
                                       std::time_t dateFrom,
                                       std::time_t dateTo,
                                       OnCountReceive hook) const = 0;

    /// erase message
    virtual void syncErase(const std::list<Mid> & mids,
                           OnUpdate hook) const = 0;

    /// erase message
    virtual void syncForceErase(const std::list<Mid> & mids,
                                OnUpdate hook) const = 0;

    /// add labels to messages
    virtual void syncAddLabels(const std::list<Label> & labels,
                               const std::list<Mid> & mids,
                               OnUpdate hook) const = 0;
    /// add labels to messages
    virtual void syncAddLabelsByThreads(const std::list<Label> & labels,
                               const std::list<Tid> & tids,
                               OnUpdateMessages hook) const = 0;

    /// remove label from messages
    virtual void syncRemoveLabels(const std::list<Label> & labels,
                                  const std::list<Mid> & mids,
                                  OnUpdate hook) const = 0;

    /// remove label from messages
    virtual void syncRemoveLabelsByThreads(const std::list<Label> & labels,
                                  const std::list<Tid> & tids,
                                  OnUpdateMessages hook) const = 0;

    /// change labels of messages
    virtual void syncChangeLabels(const std::list<Mid>& mids,
                                  const std::list<Label>& labelsToAdd,
                                  const std::list<Label>& labelsToDel,
                                  OnUpdate hook) const = 0;

    /// get messages by set of ids
    virtual void syncEntriesByIds(const std::list<Mid>& mids,
            Hook<Sequence<Envelope>> handler) const = 0;

    /// insert new message entry
    virtual void syncInsertEntry(Envelope envelope, MimeParts mime,
                                 ThreadMetaProvider p,
                                 SaveOptions saveOptions,
                                 OnUpdateEnvelope hook) const = 0;

    virtual void syncUpdateEntry(const Envelope& oldEnvelope,
            Envelope envelope, MimeParts mime, ThreadMetaProvider p, bool ignoreDuplicates,
            OnUpdateEnvelope hook) const = 0;

    virtual void syncResetFreshCounter(OnUpdate handler) const = 0;

    virtual void syncMoveMessages(const Fid& destFolder,
                                  const std::optional<Tab::Type>& destTab,
                                  const std::list<Mid>& mids,
                                  OnUpdate hook) const = 0;

    virtual void syncCopyMessages(const Fid& dstFid,
                                  const std::vector<Mid>& mids,
                                  OnCopy hook) const = 0;

    virtual void syncUpdateStatus(const std::list<Mid>& mids,
                                  Envelope::Status status,
                                  OnUpdate hook) const = 0;

    virtual void syncGetMidsByFolder(const Fid& fid,
                                     OnMidsReceive handler) const = 0;
    virtual void syncGetMidsRangeByFolder(const Fid& fid, const std::optional<Mid>& fromMid, size_t count,
                                          OnMidsReceive handler) const = 0;
    virtual void syncGetMidsByFolderWithoutStatus(const Fid& fid,
                                                  Envelope::Status excludeStatus,
                                                  OnMidsReceive handler) const = 0;
    virtual void syncGetMidsRangeByFolderWithoutStatus(const Fid& fid,
                                                       Envelope::Status excludeStatus,
                                                       const std::optional<Mid>& fromMid,
                                                       std::size_t count,
                                                       OnMidsReceive handler) const = 0;

    virtual void syncGetMidsByTab(const Tab::Type& tab,
                                  OnMidsReceive handler) const = 0;
    virtual void syncGetMidsByTabWithoutStatus(const Tab::Type& tab,
                                               Envelope::Status excludeStatus,
                                               OnMidsReceive handler) const = 0;

    virtual void syncGetAttachments(const Mid& mid,
                                    OnAttachmentsReceive handler) const = 0;

    virtual void syncGetMessageStIds(const Mids& mids,
                                     OnMidsWithStidsReceive hook) const = 0;
    virtual void syncGetWindatMessageStId(const Mid& mid, const Hid& hid,
                                          OnStIdReceive hook) const = 0;

    virtual void syncGetMimes(const Mids& mids, OnMidsWithMimes hook) const = 0;
    virtual void syncGetMimesWithDeleted(const Mids& mids, OnMidsWithMimes hook) const = 0;
    virtual void syncGetWindatMimes(const Mids& mids, OnMidsWithMimes hook) const = 0;

    virtual void syncGetMimesWithAttaches(const Mids& mids, OnMidsWithMimesAndAttaches hook) const = 0;
    virtual void syncGetMimesWithAttachesWithDeleted(const Mids& mids, OnMidsWithMimesAndAttaches hook) const = 0;

    virtual void syncCheckDuplicates(const Envelope&, CheckDuplicates) const = 0;

    virtual void syncGetMidsByConditions(const EnvelopesQueryMidsByConditions& conds,
                                         OnMidsReceive handler) const = 0;
    virtual void syncGetMidsByConditionsWithoutStatus(const EnvelopesQueryMidsByConditions& conds,
                                                      Envelope::Status excludeStatus,
                                                      OnMidsReceive handler) const = 0;

    virtual void syncDeleteFromStorage(const Stid& stid, OnExecute handler) const = 0;

    virtual uint64_t syncThreadsByTids(const TidVector& tids,
                                       OnEnvelopeReceive handler) const = 0;

    virtual uint64_t syncThreadsWithLabel(const Lid& label,
                                          size_t from,
                                          size_t count,
                                          OnEnvelopeReceive handler) const = 0;
    virtual void syncThreadsInFolderWithoutLabels(const Fid& folder,
                                                  const std::list<Lid>& labels,
                                                  size_t from,
                                                  size_t count,
                                                  OnEnvelopeReceive handler) const = 0;
    virtual void syncThreadsInFolderWithLabels(const Fid& folder,
                                               const std::list<Lid>& labels,
                                               size_t from,
                                               size_t count,
                                               OnEnvelopeReceive handler) const = 0;
    virtual void syncMessagesInFolderWithoutLabels(const Fid& folder,
                                                   const std::list<Lid>& labels,
                                                   size_t from,
                                                   size_t count,
                                                   const EnvelopesSorting& order,
                                                   OnEnvelopeReceive handler) const = 0;
    virtual void syncMessagesInFolderWithLabels(const Fid& folder,
                                                const std::list<Lid>& labels,
                                                size_t from,
                                                size_t count,
                                                const EnvelopesSorting& order,
                                                OnEnvelopeReceive handler) const = 0;

    virtual void syncGetMessageId(const Mid& mid,
                                  OnRfcMessageId handler) const = 0;
    virtual void syncGetFreshCounter(OnCountReceive handler) const = 0;
    virtual void syncGetAttachesCounters(OnAttachesCounters handler) const = 0;

    virtual void syncGetMidsWithoutStatus(const Mids& mids,
                                          Envelope::Status excludeStatus,
                                          OnMidsReceive handler) const = 0;

    virtual void syncGetMidsByTidsAndLids(const Tids& tids, const Lids& lids,
            OnMidsReceive handler) const = 0;

    virtual void syncGetByMessageId(const Fid& fid, const RfcMessageId& messageId,
                                    OnMidsAndImapIds handler) const = 0;

    virtual void syncGetByMessageId(const RfcMessageId& messageId,
                                    const macs::FidVec& excludedFids,
                                    OnMidsReceive handler) const = 0;

    virtual void syncEnvelopesInFolderWithMimes(size_t from, size_t count, const EnvelopesSorting& order,
                                                const Fid& fid, OnEnvelopeWithMimeReceive handler) const = 0;

    virtual void syncEnvelopesByMidsWithMimes(const MidList& mids, OnEnvelopeWithMimeReceive handler) const = 0;

    virtual void syncMidsByThreadAndWithSameHeaders(const Tid& threadId, OnMidsReceive handler) const = 0;

    virtual void syncMidsByHdrDateAndMessageIdPairs(const EnvelopesQueryMidsByHdrPairs& hdrPairs,
                                                    OnMidsReceive handler) const = 0;

    virtual void syncEnvelopesInFolderWithMimesByChunks(size_t minImapId, size_t maxImapId, size_t chunkSize,
                                                        const Fid& fid, OnEnvelopeWithMimeChunkReceive handler) const = 0;

    virtual void syncGetDeletedMessages(size_t from, size_t count,
                                        OnEnvelopeReceive handler) const = 0;

    virtual void syncGetDeletedMessagesInInterval(size_t from, size_t count,
                                                  std::time_t dateFrom, std::time_t dateTo,
                                                  OnEnvelopeReceive handler) const = 0;

    virtual void syncGetNewCountInFolderWithLabels(const Fid& fid,
                                                   const std::list<Lid>& labels,
                                                   size_t limit,
                                                   OnCountReceive handler) const = 0;

    virtual void syncGetNewCountInFolderWithoutLabels(const Fid& fid,
                                                      const std::list<Lid>& labels,
                                                      size_t limit,
                                                      OnCountReceive handler) const = 0;

    virtual void syncEnvelopesQueryInTab(const EnvelopesQueryInTab::Params& params,
                                         OnEnvelopeReceive handler) const = 0;

    template <typename ... Envelope>
    EnvelopeFactory getEnvelopeFactory(Envelope&& ... envelope) const {
        return EnvelopeFactory(std::forward<Envelope>(envelope)...);
    }

    virtual void syncCountNewMessagesWithLids(const std::list<Lid>& labels,
                                              size_t limit,
                                              OnCountReceive handler) const = 0;

    virtual void syncGetMidsByLabel(const Label& label, OnMidsReceive handler) const = 0;

private:
    void getByIdInternal(const Mid&, Hook<Envelope>) const;
    void markEnvelopesInternal(const std::list<Mid>&, const std::list<Label>&,
                            OnUpdate) const;
    void markEnvelopesByThreadsInternal(const std::list<Tid>&,
                            const std::list<Label>&, OnUpdate) const;
    void unmarkEnvelopesInternal(const std::list<Mid>&, const std::list<Label>&,
                            OnUpdate) const;
    void unmarkEnvelopesByThreadsInternal(const std::list<Tid>&,
                            const std::list<Label>&, OnUpdate) const;
    void removeInternal(const std::list<Mid>&, bool, OnUpdate) const;
    void resetFreshCounterInternal(OnUpdate) const;
    void moveMessagesInternal(const Fid&, const std::optional<Tab::Type>&,
                              const std::list<Mid>&, OnUpdate) const;
    void copyMessagesInternal(const Fid&, const std::vector<Mid>&, OnCopy) const;
    void updateStatusInternal(const std::list<Mid>&, Envelope::Status,
                          OnUpdate) const;
    void changeEnvelopesLabelsInternal(const std::list<Mid>&,
                                       const std::list<Label>&,
                                       const std::list<Label>&,
                                       OnUpdate) const;
    void moveMessagesToTabInternal(const std::optional<Tab::Type>&,
                                   const std::list<Mid>&,
                                   OnUpdate) const;

    enum class IdsType { mids, tids };
    static std::string toString(IdsType v) {
        switch(v) {
            case IdsType::mids : return "mids";
            case IdsType::tids : return "tids";
        }
        return "unknown";
    }

    template<typename OperationType, typename... ArgsT>
    void logOperation(ArgsT&& ... args) const {
        if( journal.get() ) {
            journal->logOperation<OperationType>(std::forward<ArgsT>(args)...);
        }
    }
    std::string getUpdateLabelsState(UserJournal::Operation operation, IdsType idsType,
            const std::list<std::string>& ids, const std::list<Label>& labels) const;
    void saveInternal( Envelope envelope, MimeParts mime, ThreadMetaProvider p,
            SaveOptions saveOptions, OnUpdateEnvelope hook ) const;
    void updateInternal( const Envelope & oldEnvelope, Envelope newEnvelope, MimeParts mime,
            ThreadMetaProvider p, bool ignoreDuplicates, OnUpdateEnvelope hook ) const;
    UserJournalPtr journal;

    template <typename LabelsRange>
    static std::list<Lid> lids(const LabelsRange& labels) {
        auto lids = labels | boost::adaptors::transformed([](auto& l){
            return l.lid();
        });
        return {std::begin(lids), std::end(lids)};
    }
};

typedef std::shared_ptr<EnvelopesRepository> EnvelopesRepositoryPtr;

} // namespace macs

