#pragma once
/*
 * @file:   errors.h
 * @author: nordsturm
 *  © Yandex LLC.
 */

#include <macs/types.h>

namespace macs {

struct MacsException: public std::runtime_error {
    MacsException(const std::string& msg)
        : std::runtime_error(msg)
    {}
};

struct MailboxLockTimeoutException: public MacsException {
    MailboxLockTimeoutException(const std::string& msg)
        : MacsException(msg)
    {}
};

struct ReadOnlyException: public MacsException {
    ReadOnlyException(const std::string& msg)
        : MacsException(msg)
    {}
};

struct ParamsException: public MacsException {
    ParamsException(const std::string& msg)
        : MacsException(msg)
    {}
};

struct UserNotInitialized: public MacsException {
    UserNotInitialized(const std::string& msg)
        : MacsException(msg)
    {}
};

namespace error {

enum Errors {
    ok,
    logic,
    input,
    end,
    duplicateLabelSymbol,
    noSuchLabel,
    invalidArgument,
    cantModifyLabel,
    userNotInitialized,
    noSuchFolder,
    cantModifyFolder,
    folderIsNotEmpty,
    folderAlreadyExists,
    folderCantBeParent,
    duplicateMessage,
    databaseLockFailed__removed, // removed from this enum
    databaseReadOnly__removed, // removed from this enum
    noSuchMessage,
    foldersLimitExceeded,
    duplicateLabelNameType,
    noSuchTab,
    noSettings,
    stickerDateOutOfInterval
};

} // namespace error
} // namespace macs

namespace boost {
namespace system {

template <>
struct is_error_code_enum<macs::error::Errors> : std::true_type {};

} // namespace system
} // namespace boost

namespace macs {
namespace error {

class Category : public error_category {
public:
    const char* name() const noexcept override {
        return "macs::error::Category";
    }

    std::string message(int v) const override {
        switch(Errors(v)) {
            case ok :
                return "no error";
            case logic :
                return "logic error";
            case input :
                return "invalid input";
            case end :
                return "end of data";
            case duplicateLabelNameType :
                return "duplicate label name type constraint violation";
            case duplicateLabelSymbol :
                return "duplicate label symbol constraint violation";
            case noSuchLabel :
                return "no such label";
            case invalidArgument :
                return "invalid argument";
            case cantModifyLabel :
                return "can not modify label";
            case userNotInitialized :
                return "user is not initialized in the database";
            case noSuchFolder :
                return "no such folder";
            case cantModifyFolder :
                return "can not modify folder";
            case folderIsNotEmpty :
                return "folder is not empty";
            case folderAlreadyExists :
                return "folder already exists";
            case folderCantBeParent :
                return "folder can not be parent";
            case duplicateMessage :
                return "message already exist";
            case databaseLockFailed__removed:
                return "databaseLockFailed__removed";
            case databaseReadOnly__removed:
                return "databaseReadOnly__removed";
            case noSuchMessage:
                return "no such message";
            case foldersLimitExceeded:
                return "folders limit exceeded";
            case noSuchTab:
                return "no such tab";
            case noSettings:
                return "failed get settings";
            case stickerDateOutOfInterval:
                return "sticker's date is out of allowed interval";
        }
        return "unknown error";
    }
};

inline const error_category& getCategory() {
    static Category instance;
    return instance;
}

inline error_code::base_type make_error_code(Errors e) {
    return error_code::base_type(static_cast<int>(e), getCategory());
}

// inline error_code::error_condition make_error_condition(Errors e) {
//     return error_code::error_condition(static_cast<int>(e), getCategory());
// }

} // namespace error

} // namespace macs
