#pragma once

#include <string>
#include <list>
#include <set>
#include <sstream>
#include <boost/algorithm/string.hpp>
#include <macs/revision.h>
#include <macs/errors.h>
#include <macs/data/symbols.h>
#include <macs/detail/strutils.h>
#include <user_journal/enumeration.h>
#include <boost/assign.hpp>

namespace macs {

using std::string;
using FidList = std::list<string>;

extern const std::string pathSeparator;

std::string normalizeAndVerifyFolderName(const std::string& rawName);
std::string escapeFolderName(const std::string& name, const std::string& separator = pathSeparator);
std::vector<std::string> parseFolderPath(std::string path, const std::string& separator = pathSeparator);
std::string joinFolderPath(const std::vector<std::string>& nodes, const std::string& separator = pathSeparator);

#define GETTER(NAME,TYPE) \
    TYPE NAME() const { \
        return NAME##_; \
    }

class Folder {
public:
    class Symbol : public MultiIndexDict<Symbol> {
    public:
        static const Symbol inbox;
        static const Symbol sent;
        static const Symbol trash;
        static const Symbol spam;
        static const Symbol drafts;
        static const Symbol outbox;
        static const Symbol archive;
        static const Symbol template_;
        static const Symbol discount;
        static const Symbol unsubscribe;
        static const Symbol zombie_folder;
        static const Symbol pending;
        static const Symbol hidden_trash;
        static const Symbol restored;
        static const Symbol reply_later;
        static const Symbol none;

        static bool isChangeble(const Folder::Symbol & symbol);

        static const Symbol & defValue() {
            return none;
        }
    private:
        Symbol(const int code, const std::string & title) : MultiIndexDict(code, title) {}
    };

    using SymbolSet = std::set<Symbol>;

    class Type : public MultiIndexDict<Type> {
    public:
        static const Type user;
        static const Type system;

        static const Type & defValue() {
            return user;
        }
    private:
        Type(const int code, const std::string & title) : MultiIndexDict(code, title) {}
    };

    using Name = std::string;
    class Path {
    private:
        using Value = std::vector<Name>;

    public:
        static constexpr std::size_t maxSize = 32;

        using const_iterator = Value::const_iterator;
        using iterator = const_iterator;
        using value_type = Value::value_type;
        using size_type = Value::size_type;

        Path() = default;

        template <typename InputIterator>
        explicit Path(const InputIterator& begin, const InputIterator& end)
            : value_(normalize(Value{begin, end})){
        }

        template <typename Range>
        explicit Path(const Range& range)
            : Path(range.begin(), range.end()) {
        }

        explicit Path(const std::string& jointPath) {
            std::vector<std::string> splited = parseFolderPath(jointPath);
            value_ = normalize(splited);
        }

        const_iterator begin() const { return value_.begin(); }

        const_iterator end() const { return value_.end(); }

        size_type size() const { return value_.size(); }

        bool empty() const { return value_.empty(); }

        std::string toString() const;

        bool operator ==(const macs::Folder::Path& other) const {
            return value_ == other.value_;
        }

        std::string name() const {
            return empty() ? "" : value_.back();
        }

    private:
        Value value_;

        static Value normalize(const Value& value);
    };

    static const std::string noParent;

    bool isSystem() const;

    GETTER(fid, const string &)
    GETTER(name, const string &)
    GETTER(type, const Folder::Type &)
    const Symbol & symbolicName() const {
        return symbol_;
    }
    bool isSpam() const {
        return symbolicName()==Symbol::spam;
    }
    bool isTrash() const {
        return symbolicName()==Symbol::trash;
    }
    bool isChangeble() const {
        return Folder::Symbol::isChangeble(symbolicName());
    }

    GETTER(bytes, size_t)
    GETTER(messagesCount, size_t)
    GETTER(newMessagesCount, size_t)
    GETTER(recentMessagesCount, size_t)
    GETTER(unvisited, bool)

    GETTER(position, size_t)
    GETTER(parentId, const string &)
    GETTER(creationTime, const string &)
    GETTER(subscribedForSharedFolder, bool)

    GETTER(revision, Revision)
    GETTER(imapUidNext, uint64_t)
    GETTER(imapUidValidity, uint64_t);
    GETTER(imapFirstUnseen, uint64_t);
    GETTER(pop3On, bool)
    GETTER(isThreadable, bool)

    static std::size_t maxFolderNameLength(void) {
        return 80;
    }

protected:
    string fid_;
    string name_;

    size_t bytes_ = 0;
    size_t messagesCount_ = 0;
    size_t newMessagesCount_ = 0;
    size_t recentMessagesCount_ = 0;
    bool unvisited_ = false;

    Symbol symbol_ = Symbol::none; ///< symbolic name for this folder
    Type type_ = Type::user; ///< type of folder

    // additional
    string parentId_ = Folder::noParent;
    size_t position_ = 0;
    string creationTime_;
    bool subscribedForSharedFolder_ = false;
    friend class FolderFactory;

    Revision revision_ = NULL_REVISION;
    uint64_t imapUidNext_ = 0;
    uint64_t imapUidValidity_ = 0;
    uint64_t imapFirstUnseen_ = 0;
    bool pop3On_ = false;
    bool isThreadable_ = true;


    struct __ArchivationType{
        enum Enum {
            unknown,
            archive,
            clean
        };

        using Map = user_journal::Enum2String<Enum>::Map;

        void fill(Map & map) const{
            boost::assign::insert(map)
                ( archive, "archive" )
                ( clean, "clean" );
        }

        using Filler = __ArchivationType;
    };

public:
    using ArchivationType = user_journal::Enumeration<__ArchivationType>;
};

typedef std::set<Folder::Symbol> SymbolSet;

#undef GETTER

inline bool Folder::isSystem() const {
    return type_ == Type::system;
}

std::string getFolderDisplayName(const std::string& path);

} // namespace macs

