#pragma once

#include <macs/folder.h>
#include <macs/folder_factory.h>
#include <map>

namespace macs {

class FoldersRepository;

inline bool isZeroOrEmpty(const std::string& id) {
    return id == "" || id == "0";
}

inline error_code checkCanEraseFolderCascade(const Folder& folder) {
    if (folder.isSystem()) {
        return error_code(error::cantModifyFolder, "can't remove system folder " + folder.fid());
    }
    return error_code();
}

using FoldersMap = std::map<Fid, const Folder>;

class FolderSet {
public:
    static constexpr size_t MAX_FOLDERS_NUMBER = 1024;
    static constexpr size_t MAX_FOLDERS_DEPTH = Folder::Path::maxSize;

    using Name = std::string;
    using size_type = FoldersMap::size_type;
    using const_iterator = FoldersMap::const_iterator;
    using value_type = const_iterator::value_type;
    using iterator = const_iterator;

    FolderSet() = default;

    FolderSet(FoldersMap data)
        : FolderSet(std::make_shared<FoldersMap>(std::move(data))) {
    }

    FolderSet(std::shared_ptr<const FoldersMap> data)
        : data(std::move(data)) {
    }

    const_iterator begin() const {
        return data ? data->begin() : emptyMap.cend();
    }

    const_iterator end() const {
        return data ? data->end() : emptyMap.cend();
    }

    bool empty() const {
        return data ? data->empty() : true;
    }

    size_type size() const {
        return data ? data->size() : 0;
    }

    const_iterator find(const Fid& fid) const {
        return data ? data->find(fid) : emptyMap.cend();
    }

    const_iterator find(const Name& name, const Fid& parentId) const;

    const_iterator find(const Folder::Path& path) const;

    const_iterator find(const Folder::Symbol & symbol) const {
        return std::find_if(begin(), end(),
                [&symbol](const value_type &v) { return v.second.symbolicName() == symbol; } );
    }

    template <typename ... Keys>
    bool exists(Keys&& ... keys) const {
        return find(std::forward<Keys>(keys)...) != end();
    }

    template <typename ... Keys>
    const Folder& at(Keys&& ... keys) const {
        const auto it = find(std::forward<Keys>(keys)...);
        if (it == end()) {
            throw system_error(error_code(error::noSuchFolder),
                    "access to nonexistent folder '"
                    + printKeys(std::forward<Keys>(keys)...) + "'");
        }
        return it->second;
    }

    template <typename ... Keys>
    const Fid& fid(Keys&& ... keys) const {
        const auto it = find(std::forward<Keys>(keys)...);
        static Fid empty;
        return it != end() ? it->second.fid() : empty;
    }

    std::vector<Folder> getChildren(const Folder& folder) const {
        std::vector<Folder> res;
        for (const auto &i : *this) {
            if (i.second.parentId() == folder.fid()) {
                res.push_back(i.second);
            }
        }
        return res;
    }

    std::vector<Folder> getChildrenRecursive(const Folder& folder) const {
        const auto children = getChildren(folder);
        auto res = children;
        for (const auto& child: children) {
            const auto grandChildren = getChildrenRecursive(child);
            res.insert(res.end(), grandChildren.begin(), grandChildren.end());
        }
        return res;
    }

    error_code checkCanCreateFolder(const Name& name, const Fid& parent, bool withSymbol) const {
        if (exists(name, parent)) {
            return error_code(error::folderAlreadyExists,
                    "can't create folder with name \""
                                + name + "\" parent " + parent);
        }

        return checkCanGetOrCreateFolder(parent, withSymbol);
    }

    error_code checkCanGetOrCreateFolder(const Fid& parent, bool withSymbol) const {
        if (!parent.empty() && parent != "0") {
            if (!exists(parent)) {
                return error_code(error::noSuchFolder,
                    "can't create folder with parent " + parent);
            }
        }

        if (!canBeParent(parent)) {
            return error_code(error::folderCantBeParent,
                    "can't create folder with parent " + parent);
        }

        return checkFolderLimits(parent, MAX_FOLDERS_NUMBER, MAX_FOLDERS_DEPTH, withSymbol);
    }

    error_code checkFolderLimits(const Fid& parent, size_t maxFoldersNumber, size_t maxFoldersDepth, bool withSymbol) const {
        if (!withSymbol && size() >= maxFoldersNumber) {
            return error_code(error::foldersLimitExceeded, "folders number limit exceeded");
        } else if (getDepth(parent) >= maxFoldersDepth) {
            return error_code(error::foldersLimitExceeded, "folders depth limit exceeded");
        }
        return error_code();
    }

    error_code checkCanRenameFolder(const Folder& folder, const Folder& original) const {
        if (original.isSystem()) {
            return error_code(error::cantModifyFolder,
                    "can't rename system folder " + folder.fid());
        }
        if (exists(folder.name(), folder.parentId())) {
            return error_code(error::folderAlreadyExists,
                    "can't rename folder " + folder.fid() +
                    " with name \"" + folder.name() +"\"");
        }
        if (folder.name().empty()) {
            return error_code(error::invalidArgument,
                    "can't rename folder " + folder.fid() +
                    " with empty name");
        }
        if (getFolderDisplayName(folder.name()).length() > Folder::maxFolderNameLength()) {
            return error_code(error::invalidArgument,
                    "can't rename folder " + folder.fid() +
                    " name is too long");
        }
        return error_code();
    }

    error_code checkCanMoveFolder(const Folder& folder, const Folder& original) const {
        if (original.isSystem()) {
            return error_code(error::cantModifyFolder,
                    "can't move system folder " + folder.fid());
        }
        if (folder.parentId() == folder.fid()) {
            return error_code(error::folderCantBeParent,
                    "can't move folder " + folder.fid() + " to itself");
        }
        if (!canBeParent(folder.parentId())) {
            return error_code(error::folderCantBeParent,
                    "can't move folder " + folder.fid() + " to " + folder.parentId());
        }
        if (exists(folder.name(), folder.parentId())) {
            return error_code(error::folderAlreadyExists,
                    "can't move folder " + folder.fid() + " with name \"" +
                    folder.name() + "\" to " + folder.parentId());
        }
        return error_code();
    }

    error_code checkCanUpdateFolder(const Folder& folder, const Folder& original) const {
        if (folder.name() != original.name()) {
            error_code err = checkCanRenameFolder(folder, original);
            if (err) {
                return err;
            }
        }

        if (folder.parentId() != original.parentId()) {
            return checkCanMoveFolder(folder, original);
        }
        return error_code();
    }

    error_code checkCanEraseFolder(const Folder &folder) const {
        auto e = checkCanEraseFolderCascade(folder);
        if(e) {
            return e;
        }
        if (folder.messagesCount() > 0) {
            return error_code(error::folderIsNotEmpty,
                    "can not delete folder " + folder.fid() + " with messages");
        }
        if (getChildren(folder).size() > 0) {
            return error_code(error::folderIsNotEmpty,
                     "can not delete folder " + folder.fid() + " with subfolders");
        }
        return error_code();
    }

    Folder::Path getPath(const Folder& folder) const {
        std::vector<Folder::Name> revertPath = {folder.name()};
        std::string parentId = folder.parentId();
        while(!isZeroOrEmpty(parentId)) {
            auto parent = at(parentId);
            revertPath.push_back(parent.name());
            parentId = parent.parentId();
        }
        return Folder::Path(revertPath.rbegin(), revertPath.rend());
    }

private:
    static const SymbolSet& childlessFolderSymbols() {
        static const SymbolSet symbols = {
              Folder::Symbol::outbox
            , Folder::Symbol::spam
            , Folder::Symbol::archive
            , Folder::Symbol::template_
            , Folder::Symbol::discount
            , Folder::Symbol::pending
            , Folder::Symbol::hidden_trash
            , Folder::Symbol::reply_later
        };
        return symbols;
    }

    bool canBeParent(const Folder& folder) const {
        return !childlessFolderSymbols().count(folder.symbolicName());
    }

    bool canBeParent(const Fid& parentFid) const {
        if (isZeroOrEmpty(parentFid)) {
            return true;
        }
        return exists(parentFid) && canBeParent(at(parentFid));
    }

    size_t getDepth(Fid fid) const {
        size_t cnt = 0;
        while(!fid.empty() && fid != "0") {
            fid = find(fid)->second.parentId();
            ++cnt;
        }
        return cnt;
    }

    std::string printKeys(const Folder::Symbol & symbol) const {
        return symbol.title();
    }

    template <typename FirstKey, typename ... Keys>
    std::string printKeys(FirstKey&& firstKey, Keys&& ...) const {
        std::ostringstream s;
        s << firstKey;
        return s.str();
    }

    std::shared_ptr<const FoldersMap> data;

    static const FoldersMap emptyMap;
};

inline FolderSet::const_iterator FolderSet::find(const FolderSet::Name& name,
        const Fid& parentId) const {
    const auto matchName = [&name](const Folder& f) {
        return f.name() == name;
    };

    const auto matchParent = [&parentId](const Folder& f) {
        return (f.parentId() == parentId) ||
            (isZeroOrEmpty(f.parentId()) && isZeroOrEmpty(parentId));
    };

    return std::find_if(begin(), end(), [=](const value_type &v) {
        return matchName(v.second) && matchParent(v.second);
    });
}

inline FolderSet::const_iterator FolderSet::find(const Folder::Path& path) const {
    return std::find_if(begin(), end(), [this, &path] (const value_type &v) {
        return getPath(v.second) == path;
    });
}

inline Folder resetFolderParent( Folder folder ) {
    FolderFactory factory(std::move(folder));
    factory.parentId("0");
    return factory.product();
}

inline Folder setFolderParent( Folder folder, const Folder& parent) {
    FolderFactory factory(std::move(folder));
    factory.parentId(parent.fid());
    return factory.product();
}

inline FolderSet makeFoldersWithoutHidden(FolderSet foldersWithHidden) {
    FoldersMap folders(foldersWithHidden.begin(), foldersWithHidden.end());
    auto it = std::find_if(folders.begin(), folders.end(),
            [](const auto &v) { return v.second.symbolicName() == Folder::Symbol::hidden_trash; } );
    if (it != folders.end()) {
        folders.erase(it);
    }
    return FolderSet(std::move(folders));
}

} // namespace macs
