#pragma once

#include <map>

#include <macs/types.h>
#include <macs/folder_factory.h>
#include <macs/mailbox_space_info.h>
#include <macs/hooks.h>
#include <boost/scoped_ptr.hpp>
#include <macs/user_journal.h>
#include <macs/folder_set.h>
#include <macs/detail/cache.h>
#include <boost/optional.hpp>
#include <boost/uuid/random_generator.hpp>
#include <boost/uuid/uuid_io.hpp>
#include <macs/io.h>

namespace macs {

/// Folders access and manipulation class
class FoldersRepository : public std::enable_shared_from_this<FoldersRepository> {
public:
    FoldersRepository() = default;
    FoldersRepository(UserJournalPtr journal) : journal(journal) { }

    virtual ~FoldersRepository() = default;

    MailboxSpaceInfo getMailboxSpaceInfo() const;

    template <typename Handler = io::sync_context>
    auto getMailboxRevision(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnRevisionReceive> init(handler);
        syncGetMailboxRevision(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getAllFolders(Handler handler = io::use_sync) const {
        return getAllFoldersImpl(FoldersCacheType::withoutHidden, handler);
    }

    template <typename Handler = io::sync_context>
    auto getAllFoldersWithHidden(Handler handler = io::use_sync) const {
        return getAllFoldersImpl(FoldersCacheType::withHidden, handler);
    }

    template <typename Handler = io::sync_context>
    auto getAllPop3Folders(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnPop3Folders> init(handler);
        syncGetPop3Folders(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto subscribeToSharedFolders(const std::string& suid,
                                    Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnFoldersMap> init(handler);
        subscribeToSharedFoldersInternal(suid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto unsubscribeFromSharedFolders(const std::string& suid,
                                    Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnFoldersMap> init(handler);
        syncUnsubscribeFromSharedFolders(suid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto createFolder(const std::string& name,
                      const Fid& parent,
                      const Folder::Symbol& symbol,
                      Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateFolder> init(handler);
        createFolderInternal(name, parent, symbol, init.handler);
        return init.result.get();
    }
    template <typename Handler = io::sync_context>
    auto createFolder(const std::string& name,
                      const Fid& parent,
                      Handler handler = io::use_sync) const {
        return createFolder(name, parent, Folder::Symbol::none, std::move(handler));
    }
    template <typename Handler = io::sync_context>
    auto createFolder(const std::string& name, Handler handler = io::use_sync) const {
        return createFolder(name, Folder::noParent, std::move(handler));
    }

    template <typename Handler = io::sync_context>
    auto getOrCreateFolder(const std::string& name,
                           const Fid& parent,
                           const Folder::Symbol& symbol,
                           Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateFolder> init(handler);
        getOrCreateFolderInternal(name, parent, symbol, init.handler);
        return init.result.get();
    }
    template <typename Handler = io::sync_context>
    auto getOrCreateFolder(const std::string& name,
                           const Fid& parent,
                           Handler handler = io::use_sync) const {
        return getOrCreateFolder(name, parent, Folder::Symbol::none, std::move(handler));
    }
    template <typename Handler = io::sync_context>
    auto getOrCreateFolder(const std::string& name, Handler handler = io::use_sync) const {
        return getOrCreateFolder(name, Folder::noParent, std::move(handler));
    }

    template <typename Handler = io::sync_context>
    auto createFolderByPath(const Folder::Path& path, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateFolder> init(handler);
        createFolderByPathInternal(path, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateFolder(Folder folder, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateFolder> init(handler);
        updateFolderInternal(std::move(folder), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateFolderToPath(FolderFactory factory, Folder::Path path, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateFolder> init(handler);
        updateFolderToPathInternal(std::move(factory), std::move(path), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updatePosition(Folder folder, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateFolder> init(handler);
        updatePositionInternal(std::move(folder), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto erase(const Folder & folder, Handler handler = io::use_sync) const{
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        eraseInternal(std::move(folder), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto eraseCascade(const Folder & folder, Handler handler = io::use_sync) const{
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        eraseCascadeInternal(std::move(folder), init.handler);
        return init.result.get();
    }

    /// move data from folder to another folder
    template <typename Handler = io::sync_context>
    auto moveAssociatedData(const Folder & from, const Folder & to,
            Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        moveAssociatedDataInternal(from, to, init.handler);
        return init.result.get();
    }

    void resetFoldersCache() const {
        cache[FoldersCacheType::withHidden]->reset();
        cache[FoldersCacheType::withoutHidden]->reset();
    }

    template <typename Handler = io::sync_context>
    auto setSymbol(const Folder & folder, const Folder::Symbol & symbol,
            Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        setSymbolInternal(folder, symbol, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto resetSymbol(const Folder & folder, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        resetSymbolInternal(folder, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto resetUnvisited(const Folder & folder, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        resetUnvisitedInternal(folder, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto moveToTop(Folder folder, Handler handler = io::use_sync) const {
        return updateFolder(resetFolderParent(std::move(folder)), std::move(handler));
    }
    template <typename Handler = io::sync_context>
    auto moveToParent(Folder folder, const Folder & parent, Handler handler = io::use_sync) const {
        return updateFolder(setFolderParent(std::move(folder), parent), std::move(handler));
    }

    template <typename Handler = io::sync_context>
    auto setPop3(std::vector<Fid> fids, Handler handler = io::use_sync) const{
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        setPop3Internal(std::move(fids), init.handler);
        return init.result.get();
    }

    // !!! Deprecated method moveFolder(), use moveToXXX() instead !!!
    Folder moveFolder(const Fid& fid, const Fid& parentFid) const {
        auto folders = foldersInternal();
        if(parentFid == Folder::noParent) {
            return moveToTop(folders->at(fid));
        }
        return moveToParent(folders->at(fid), folders->at(parentFid));
    }
    //!!! Deprecated method resetUnvisited(fid), use resetUnvisited(Folder) instead !!!
    Revision resetUnvisited(const Fid& fid) const {
        auto folders = foldersInternal();
        const auto i = folders->find(fid);
        return i != folders->end() ? resetUnvisited(i->second) : NULL_REVISION;
    }
    // !!! Deprecated method getFolderFidBySymbol(), use FolderSet::getFolderFidBySymbol() instead !!!
    std::string getFolderFidBySymbol(const Folder::Symbol & symbol) const;
    // !!! Deprecated method getFolderBySymbol(), use FolderSet::getFolderBySymbol() instead !!!
    Folder getFolderBySymbol(const Folder::Symbol & symbol) const;
    // !!! Deprecated method deleteFolder(), use erase() instead !!!
    Revision deleteFolder(const Fid& fid, const bool forced = false) const {
        if (fid.empty()) {
            throw system_error(error_code(error::invalidArgument), "deleteFolder: empty fid");
        }
        const auto folder = foldersInternal()->at(fid);
        return forced ? eraseCascade(folder) : erase(folder);
    }
    // !!! Deprecated method getFolderByFid(), use FolderSet::getFolderByFid() instead !!!
    Folder getFolderByFid(const Fid& fid) const;
    // !!! Deprecated method getFolderByFid(), use FolderSet::getFolderByFid() instead !!!
    template<typename Iterator, typename Inserter>
    Inserter getFoldersByFids(Iterator first, Iterator last, Inserter inserter) const {
        const auto folders = foldersInternal();
        std::transform(first, last, inserter, [folders](const auto& i){ return folders->at(i); });
        return inserter;
    }
    // !!! Deprecated method getFolderByFid(), use FolderSet::existFolder() instead !!!
    bool existFolder(const Fid& fid) const;

    template <typename Handler = io::sync_context>
    auto setArchivationRule(const Fid& fid,
                            const Folder::ArchivationType& type,
                            uint32_t keep_days,
                            uint32_t max_size,
                            Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncSetArchivationRule(fid, type, keep_days, max_size, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto removeArchivationRule(const Fid& fid,
                               Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncRemoveArchivationRule(fid, init.handler);
        return init.result.get();
    }

protected:
    void subscribeToSharedFoldersInternal(const std::string& sharedFoldersSuid,
                                        OnFoldersMap handler) const;
    void createFolderInternal(const std::string& name,
                              const Fid& parent,
                              const Folder::Symbol& symbol,
                              OnUpdateFolder hook) const;
    void getOrCreateFolderInternal(const std::string& name,
                                   const Fid& parent,
                                   const Folder::Symbol& symbol,
                                   OnUpdateFolder hook) const;
    void createFolderByPathInternal(const Folder::Path& path, OnUpdateFolder hook) const;
    void updateFolderInternal(Folder folder, OnUpdateFolder handler) const;
    void updateFolderToPathInternal(FolderFactory factory, Folder::Path path, OnUpdateFolder hook) const;
    void updatePositionInternal(Folder folder, OnUpdateFolder handler) const;
    void eraseInternal(const Folder & folder, OnUpdate handler) const;
    void eraseCascadeInternal(const Folder & folder, OnUpdate handler) const;
    void moveAssociatedDataInternal(const Folder & from, const Folder & to,
                                OnUpdate handler) const;
    void setSymbolInternal(const Folder & folder, const Folder::Symbol & symbol,
                OnUpdate handler) const;
    void resetSymbolInternal(const Folder & folder, OnUpdate handler) const;
    void resetUnvisitedInternal(const Folder & folder, OnUpdate hook) const;
    void setPop3Internal(std::vector<Fid> fids, OnUpdate handler) const;

    virtual MailboxSpaceInfo syncGetMailboxSpaceInfo() const = 0;

    virtual void syncGetMailboxRevision(OnRevisionReceive hook) const = 0;

    virtual void syncGetFolders(OnFolders) const = 0;

    virtual void syncGetPop3Folders(OnPop3Folders) const = 0;

    // should we use entry instead of fid here?
    virtual void syncResetUnvisited(const Fid& fid, OnUpdate hook) const = 0;

    virtual void syncSubscribeToSharedFolders(const std::string& sharedFoldersSuid, OnFoldersMap) const = 0;
    virtual void syncUnsubscribeFromSharedFolders(const std::string& sharedFoldersSuid, OnFoldersMap) const = 0;

    virtual void syncCreateFolder(const std::string& name,
                                  const Fid& parent,
                                  const Folder::Symbol& symbol,
                                  OnUpdateFolder handle) const = 0;
    virtual void syncGetOrCreateFolder(const std::string& name,
                                       const Fid& parent,
                                       const Folder::Symbol& symbol,
                                       OnUpdateFolder handle) const = 0;
    virtual void syncCreateFolderByPath(Folder::Path path, OnUpdateFolder handle) const = 0;
    virtual void syncModifyFolder(const Folder& folder, OnUpdateFolder handler) const = 0;
    virtual void syncModifyFolderToPath(const Folder& folder, const Folder::Path& path, OnUpdateFolder handler) const = 0;
    virtual void syncSetPosition(const Fid& fid, size_t pos, OnUpdate handler) const = 0;

    virtual void syncEraseFolder(const Fid& fid, OnUpdate handler) const = 0;
    virtual void syncClearFolderCascade(const Fid& fid,
                                        const FolderSet & folders,
                                        OnUpdateMessages handler) const = 0;

    virtual void syncMoveAll(const Folder & src,
                             const Folder & dst,
                             OnUpdateMessages handler) const = 0;

    virtual void syncSetFolderSymbol(const Fid& fid, const Folder::Symbol & symbol,
            OnUpdate handler) const = 0;

    virtual void syncSetPop3(const std::vector<Fid>& fids,
                             const FolderSet& fs,
                             OnUpdate handler) const = 0;

    virtual void asyncSetArchivationRule(const Fid& fid,
                                         const Folder::ArchivationType& type,
                                         uint32_t keep_days,
                                         uint32_t max_size,
                                         OnUpdate handler) const = 0;

    virtual void asyncRemoveArchivationRule(const Fid& fid,
                                            OnUpdate handler) const = 0;

    FolderFactory getFolderFactory() const {
        return FolderFactory();
    }

    virtual const Folder::SymbolSet& defaultFoldersSymbols() const;

    error_code checkFoldersRange(const FolderSet& folders) const;

private:
    using FoldersCache = detail::Cache<FolderSet>;
    using CachePtr = FoldersCache::Ptr;

    void setFolderPosition(Folder& folder, const FolderSet& fs) const;
    std::vector<Folder> setFoldersInitialPosition(FoldersMap& folders) const;
    template <typename Handler>
    void updatePositionForFolders(std::vector<Folder> folders, Handler h)const;

    void onFolderCreate(Folder folder, OnUpdateFolder hook) const;

    enum class FoldersCacheType {
        withHidden,
        withoutHidden
    };

    template <typename Handler>
    auto getAllFoldersImpl(FoldersCacheType cacheType, Handler& handler) const;
    CachePtr foldersInternal() const;
    template <typename Handler>
    void foldersInternal(Handler handler) const;
    template <typename Handler>
    void foldersInternal(FoldersCacheType cacheType, Handler handler) const;

    template<typename OperationType, typename... ArgsT>
    void logOperation(ArgsT&& ... args) const {
        if( journal.get() ) {
            journal->logOperation<OperationType>(std::forward<ArgsT>(args)...);
        }
    }

    mutable std::map<FoldersCacheType, std::shared_ptr<FoldersCache>> cache = {
        {FoldersCacheType::withHidden, std::make_shared<FoldersCache>()},
        {FoldersCacheType::withoutHidden, std::make_shared<FoldersCache>()},
    };
    UserJournalPtr journal;
};

template <typename Handler>
inline auto FoldersRepository::getAllFoldersImpl(FoldersCacheType cacheType, Handler& handler) const {
    io::detail::init_async_result<Handler, OnFolders> init(handler);
    foldersInternal(cacheType, [h = init.handler](error_code e, CachePtr c) mutable {
        if(e) {
            h(std::move(e), FolderSet{});
        } else {
            h(error_code(), *c);
        }
    });
    try {
        return init.result.get();
    } catch (const system_error& e) {
        if(e.code() == error::userNotInitialized) {
            throw UserNotInitialized(e.what());
        } else {
            throw;
        }
    }
}

template <typename Handler>
inline void FoldersRepository::foldersInternal(FoldersCacheType cacheType, Handler handler) const {
    if (auto c = cache[cacheType]->get()) {
        handler(error_code(), c);
        return;
    }

    syncGetFolders([self = shared_from_this(), h = std::move(handler), cacheType]
                   (error_code e, FolderSet v) mutable {
        if (!e) {
            e = self->checkFoldersRange(v);
        }
        if (e) {
            h(std::move(e), self->cache[cacheType]->get());
        } else {
            self->cache[FoldersCacheType::withoutHidden]->set(makeFoldersWithoutHidden(v));
            self->cache[FoldersCacheType::withHidden]->set(std::move(v));
            h(std::move(e), self->cache[cacheType]->get());
        }
    });
}

template <typename Handler>
inline void FoldersRepository::foldersInternal(Handler handler) const {
    foldersInternal(FoldersCacheType::withHidden, std::move(handler));
}

inline MailboxSpaceInfo FoldersRepository::getMailboxSpaceInfo() const {
    return syncGetMailboxSpaceInfo();
}

typedef std::shared_ptr<FoldersRepository> FoldersRepositoryPtr;

template <typename Handler = io::sync_context>
macs::Folder getOrCreateFolderBySymbolWithRandomizedName(const FoldersRepository& foldersRepository,
                                                         const std::string& rawName,
                                                         const Fid& parent,
                                                         const Folder::Symbol& symbol,
                                                         bool forceRandomize,
                                                         Handler h = macs::io::use_sync) {
    static_assert(std::is_same_v<Handler, io::sync_context> ||  io_result::is_yield_context_t<Handler>::value,
                  "unsupported handler type");

    if (Folder::Symbol::none == symbol) {
        throw std::invalid_argument("strange folders symbol");
    }

    const auto folders = foldersRepository.getAllFoldersWithHidden(h);
    if (folders.exists(symbol)) {
        return folders.at(symbol);
    }

    std::string name = rawName;
    if (forceRandomize || folders.exists(name, parent)) {
        boost::uuids::random_generator generator;
        name = rawName + "_" + boost::uuids::to_string(generator());
    }

    return foldersRepository.createFolder(name, parent, symbol, h);
}

} // namespace macs
