#ifndef MACS_ENVELOPES_REPOSITORY_IMAP_H
#define MACS_ENVELOPES_REPOSITORY_IMAP_H

#include <macs/label_set.h>
#include <macs/envelopes_repository.h>
#include <user_journal/parameters/imap.h>
#include <boost/iterator/counting_iterator.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <macs/io.h>

namespace macs {

namespace jparams = user_journal::parameters;

class ImapRepository : public std::enable_shared_from_this<ImapRepository> {
public:
    enum ImapListMode {
        ImapListMode_byNum,
        ImapListMode_byUid
    };
    ImapRepository() = default;

    ImapRepository(UserJournalPtr journal) : journal(journal) {}

    virtual ~ImapRepository() = default;

    // Get mapping from uids to imap numbers
    template <typename Handler>
    auto imapGetUidMap(const ImapFolder& folder, Handler handler) const {
        io::detail::init_async_result<Handler, imap_uid_map_entry> init(handler);
        asyncImapGetUidMap(folder, init.handler);
        return init.result.get();
    }

    /// load base imap message list
    template <typename Handler>
    auto imapGetMails(const ImapFolder& folder, uint64_t low,
            uint64_t hi, ImapListMode mode, Handler handler) const {

        io::detail::init_async_result<Handler, imap_envelope_entry> init(handler);
        asyncImapGetMails(folder, low, hi, mode, init.handler);
        return init.result.get();
    }

    /// load base imap message list
    template <typename Handler>
    auto imapGetMailsChunk(const ImapFolder& folder, uint64_t low,
            uint64_t hi, uint64_t maxChunkSize, Handler handler) const {

        io::detail::init_async_result<Handler, imap_envelope_entry> init(handler);
        asyncImapGetMailsChunk(folder, low, hi, maxChunkSize, init.handler);
        return init.result.get();
    }

    /// load changed messages in folder
    template <typename Handler>
    auto imapGetChanges(const ImapFolder& folder, Handler handler) const {
        io::detail::init_async_result<Handler, imap_envelope_changes_entry> init(handler);
        asyncImapGetChanges(folder, init.handler);
        return init.result.get();
    }

    /// load messages by imap message_id
    template <typename Handler>
    auto imapGetByMessageId(const ImapFolder& folder, const RfcMessageId& messageId,
            Handler handler) const {

        io::detail::init_async_result<Handler, imap_envelope_details_entry> init(handler);
        asyncImapGetByMessageId(folder, messageId, init.handler);
        return init.result.get();
    }

    /// load imap messages details
    template <typename Handler>
    auto imapGetDetails(const ImapFolder& folder, uint64_t low, uint64_t hi,
            Handler handler) const {

        io::detail::init_async_result<Handler, imap_envelope_details_entry> init(handler);
        asyncImapGetDetails(folder, low, hi, init.handler);
        return init.result.get();
    }

    /// load imap messages details
    template <typename Handler>
    auto imapGetDetails(const ImapFolder& folder, const std::vector<uint64_t>& uids,
            Handler handler) const {

        io::detail::init_async_result<Handler, imap_envelope_details_entry> init(handler);
        asyncImapGetDetails(folder, uids, init.handler);
        return init.result.get();
    }

    /// load imap deleted messages
    template <typename Handler>
    auto imapGetDeleted(const ImapFolder& folder, Handler handler) const {
        io::detail::init_async_result<Handler, imap_envelope_entry> init(handler);
        asyncImapGetDeleted(folder, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto imapUpdateFlags(const ImapFolder& folder, const LabelSet& del,
            const LabelSet& add, const std::vector<uint64_t>& uids, Handler handler) const {

        io::detail::init_async_result<Handler, imap_envelope_oper> init(handler);
        asyncImapUpdateFlags(folder, del, add, uids,
                wrapForLog<jparams::ImapAddFlags>(uids, init.handler));
        return init.result.get();
    }

    template <typename Handler>
    auto imapDelEnvelope(const ImapFolder& folder, uint64_t uid, Handler handler) const {
        return imapDelEnvelope(folder, {uid}, std::move(handler));
    }

    template <typename Handler>
    auto imapDelEnvelope(const ImapFolder& folder, uint64_t low, uint64_t hi,
            Handler handler) const {
        boost::counting_iterator<uint64_t> begin(low);
        boost::counting_iterator<uint64_t> end(hi);
        return imapDelEnvelope(folder, {begin, end}, std::move(handler));
    }

    template <typename Handler>
    auto imapDelEnvelope(const ImapFolder& folder, const std::vector<uint64_t>& uids,
            Handler handler) const {
        io::detail::init_async_result<Handler, imap_envelope_oper> init(handler);
        asyncImapDelEnvelope(folder, uids,
                        wrapForLog<jparams::ImapDeleteMessages>(uids, init.handler));
        return init.result.get();
    }

    template <typename Handler>
    auto imapCopyEnvelope(const ImapFolder& from, const ImapFolder& to, uint64_t uid,
            Handler handler) const {
        return imapCopyEnvelope(from, to, {uid}, std::move(handler));
    }

    template <typename Handler>
    auto imapCopyEnvelope(const ImapFolder& from, const ImapFolder& to, uint64_t low,
            uint64_t hi, Handler handler) const {

        boost::counting_iterator<uint64_t> begin(low);
        boost::counting_iterator<uint64_t> end(hi);
        return imapCopyEnvelope(from, to, {begin, end}, std::move(handler));
    }

    template <typename Handler>
    auto imapCopyEnvelope(const ImapFolder& from, const ImapFolder& to,
            const std::vector<uint64_t>& uids, Handler handler) const {

        io::detail::init_async_result<Handler, imap_envelope_oper> init(handler);
        asyncImapCopyEnvelope(from, to, uids,
                        wrapForLog<jparams::ImapCopyMessages>(uids, init.handler));
        return init.result.get();
    }

    template <typename Handler>
    auto imapMoveEnvelope( const ImapFolder& from, const ImapFolder& to,
            const std::vector<uint64_t>& uids, Handler handler) const {

        io::detail::init_async_result<Handler, imap_envelope_oper> init(handler);
        asyncImapMoveEnvelope(from, to, uids,
                wrapForLog<jparams::ImapCopyMessages>(uids, init.handler));
        return init.result.get();
    }

    template <typename Handler>
    auto imapGetUnsubscribed(Handler handler) const {
        io::detail::init_async_result<Handler, OnImapUnsubscribed> init(handler);
        asyncImapGetUnsubscribed(init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto imapSubscribeFolder( const std::vector<std::string>& fullNameParts,
            Handler handler) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncImapSubscribeFolder(fullNameParts, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto imapUnsubscribeFolder( const std::vector<std::string>& fullNameParts,
            Handler handler) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncImapUnsubscribeFolder(fullNameParts, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto imapRegenerateImapId(const std::string& mid, Handler handler) const {
        io::detail::init_async_result<Handler, OnImapId> init(handler);
        asyncImapRegenerateImapId(mid, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto pop3GetFolders(const std::string &suid, Handler handler) const {
        io::detail::init_async_result<Handler, pop3_folder_entry> init(handler);
        asyncPop3GetFolders(suid, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto pop3LoadFullMessagesSingleFolder(const Fid& folderId, const Fid& spamFid,
            Handler handler) const {
        io::detail::init_async_result<Handler, pop3_full_message_entry> init(handler);
        asyncPop3LoadFullMessagesSingleFolder(folderId, spamFid, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto pop3LoadFullMessages(const std::string &suid, const Fid& spamFid,
        const FidList& fids, Handler handler) const {
        io::detail::init_async_result<Handler, pop3_full_message_entry> init(handler);
        asyncPop3LoadFullMessages(suid, spamFid, fids, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto pop3DeleteMessages(const std::list<Mid>& mids, Handler handler) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncPop3DeleteMessages(mids, init.handler);
        return init.result.get();
    }

protected:
    template <typename Range>
    static auto print(const Range& r) {
        return boost::algorithm::join(r |
                boost::adaptors::transformed(
                        [](auto& v) { return std::to_string(v);}), ",");
    }

    template <typename Oper, typename Range>
    void logOperationWithUids(const Range& uids) const {
        using namespace jparams;
        logOperation<Oper>(id::state(print(uids)), id::affected(uids.size()));
    }

    template <typename Oper>
    auto wrapForLog(const std::vector<uint64_t>& uids, imap_envelope_oper handler) const {
        auto self = shared_from_this();
        return [self, handler, uids](error_code ec, uint32_t res) {
            if(!ec) {
                self->logOperationWithUids<Oper>(uids);
            }
            handler(std::move(ec), res);
        };
    }

    // Get mapping from uids to imap numbers
    virtual void asyncImapGetUidMap(
        const ImapFolder& folder,
        imap_uid_map_entry handler) const = 0;

    /// load base imap message list
    virtual void asyncImapGetMails(const ImapFolder& folder,
        uint64_t low, uint64_t hi, ImapListMode mode,
        imap_envelope_entry handler) const = 0;

    /// load base imap message list
    virtual void asyncImapGetMailsChunk(const ImapFolder& folder,
        uint64_t low, uint64_t hi, uint64_t maxChunkSize,
        imap_envelope_entry handler) const = 0;

    /// load changed messages in folder
    virtual void asyncImapGetChanges(const ImapFolder& folder,
        imap_envelope_changes_entry handler) const = 0;

    /// load messages by imap message_id
    virtual void asyncImapGetByMessageId(
        const ImapFolder& folder, const std::string& messageId,
        imap_envelope_details_entry handler) const = 0;

    /// load imap messages details
    virtual void asyncImapGetDetails(
        const ImapFolder& folder, uint64_t low, uint64_t hi,
        imap_envelope_details_entry handler) const = 0;

    /// load imap messages details
    virtual void asyncImapGetDetails(
        const ImapFolder& folder, const std::vector<uint64_t>& uids,
        imap_envelope_details_entry handler) const = 0;

    /// load imap deleted messages
    virtual void asyncImapGetDeleted(
        const ImapFolder& folder,
        imap_envelope_entry handler) const = 0;

    virtual void asyncImapUpdateFlags(
        const ImapFolder& folder, const LabelSet& del,
        const LabelSet& add, std::vector<uint64_t> uid,
        imap_envelope_oper handler) const = 0;

    virtual void asyncImapDelEnvelope(
        const ImapFolder& folder, const std::vector<uint64_t>& uids,
        imap_envelope_oper handler) const = 0;

    virtual void asyncImapCopyEnvelope(
        const ImapFolder& from, const ImapFolder& to,
        const std::vector<uint64_t>& uids,
        imap_envelope_oper handler) const = 0;

    virtual void asyncImapMoveEnvelope(
        const ImapFolder& from, const ImapFolder& to,
        const std::vector<uint64_t>& uids,
        imap_envelope_oper handler) const = 0;

    virtual void asyncImapGetUnsubscribed(OnImapUnsubscribed) const = 0;

    virtual void asyncImapSubscribeFolder(
        const std::vector<std::string>& fullNameParts,
        OnUpdate handler) const = 0;

    virtual void asyncImapUnsubscribeFolder(
        const std::vector<std::string>& fullNameParts,
        OnUpdate handler) const = 0;

    virtual void asyncImapRegenerateImapId(
        const std::string& mid, OnImapId handler) const = 0;


    virtual void asyncPop3GetFolders(
        const std::string& suid,
        pop3_folder_entry handler) const = 0;

    virtual void asyncPop3LoadFullMessagesSingleFolder(
        const Fid& folderId,
        const Fid& spamFid,
        pop3_full_message_entry handler) const = 0;

    virtual void asyncPop3LoadFullMessages(
        const std::string& suid,
        const Fid& spamFid,
        const FidList& fids,
        pop3_full_message_entry handler) const = 0;

    virtual void asyncPop3DeleteMessages(
        const std::list<std::string>& mids,
        OnUpdate handler) const = 0;

private:
    template<typename OperationType, typename... ArgsT>
    void logOperation(ArgsT&& ... args) const {
        if( journal.get() ) {
            journal->logOperation<OperationType>(std::forward<ArgsT>(args)...);
        }
    }
    const UserJournalPtr journal;
};

typedef std::shared_ptr<ImapRepository> ImapRepositoryPtr;

}

#endif  /* MACS_ENVELOPES_REPOSITORY_IMAP_H */
