#pragma once

#include <string>
#include <list>
#include <macs/errors.h>
#include <macs/revision.h>
#include <macs/data/symbols.h>
#include <macs/detail/strutils.h>

namespace macs {

using std::string;

#define GETTER(NAME,TYPE) \
    TYPE NAME() const { \
        return NAME##_; \
    }

/** @class Label
 *  @brief Label class for marking envelopes with.
 *
 * @todo refactor CustomField
 */
class Label {
public:
    /** @class Symbol
     *  @brief Symbolic label name.
     *
     * Symbol class is a subclass because its not exist without label objects.
     */
    class Symbol : public MultiIndexDict<Symbol> {
    public:
        static const Symbol attached_label;
        static const Symbol spam_label;
        static const Symbol answered_label;
        static const Symbol recent_label;
        static const Symbol draft_label;
        static const Symbol deleted_label;
        static const Symbol forwarded_label;
        static const Symbol important_label;
        static const Symbol forMe_label;
        static const Symbol noBody_label;
        static const Symbol noAnswer_label;
        static const Symbol hasUserLabels_label;
        static const Symbol seen_label;
        static const Symbol pinned_label;
        static const Symbol postmaster_label;
        static const Symbol mulcaShared_label;
        static const Symbol imap_label;
        static const Symbol append_label;
        static const Symbol copy_label;
        static const Symbol synced_label;
        static const Symbol remindNoAnswer_label;
        static const Symbol notifyNoAnswer_label;
        static const Symbol remindMessage_label;
        static const Symbol notifyMessage_label;
        static const Symbol mute_label;
        static const Symbol delayedMessage_label;
        static const Symbol undoMessage_label;
        static const Symbol hamon_label;
        static const Symbol encrypted_label;
        static const Symbol mention_label;
        static const Symbol mention_unvisited_label;
        static const Symbol sending_failed_label;
        static const Symbol from_favorite_user_label;
        static const Symbol reply_later_started;
        static const Symbol reply_later_finished;
        static const Symbol none;

        static const Symbol & defValue() {
            return none;
        }
    private:
        Symbol(const int code, const std::string & title) : MultiIndexDict(code, title) {}
    };

    /// type of label (user/system/imap/etc)
    class Type : public MultiIndexDict<Type> {
    public:
        static const Type user;
        static const Type social;
        static const Type system;
        static const Type status;
        static const Type imap;
        static const Type threadWide;
        static const Type rpop;
        static const Type phone;
        static const Type spamDefense;
        static const Type spamDefense2;

        static const Type & defValue() {
            return user;
        }
    private:
        Type(const int code, const std::string & title) : MultiIndexDict(code, title) {}
    };

    bool isUser() const;
    bool isSystem() const;

    GETTER(lid, const string &)
    GETTER(name, const string &)
    GETTER(color, const string &)
    GETTER(type, const Label::Type &)
    const Symbol & symbolicName() const {
        return symbol_;
    }

    GETTER(newMessagesCount, size_t)
    GETTER(imapIndex, size_t)
    GETTER(creationTime, const string &)
    GETTER(revision, Revision)

    static std::size_t maxLabelNameLength(void) {
        return 512;
    }

    std::optional<size_t> realMessagesCount() const {
        return messagesCount_;
    }

    size_t messagesCount() const {
        return messagesCount_.value_or(0);
    }

    GETTER(unreadMessagesCount, std::optional<size_t>)

    Label()
        : messagesCount_(std::nullopt)
        , unreadMessagesCount_(std::nullopt)
        , newMessagesCount_(0)
        , imapIndex_(0)
        , symbol_(Symbol::none)
        , type_(Type::user)
        , revision_(NULL_REVISION)
    { }

protected:
    string lid_;
    string name_;

    std::optional<size_t> messagesCount_;
    std::optional<size_t> unreadMessagesCount_;
    size_t newMessagesCount_;
    size_t imapIndex_;

    Symbol symbol_; ///< symbolic name for this label
    Type type_; ///< type of label

    // additional
    string color_;
    string creationTime_;
    Revision revision_;
    friend class LabelFactory;
    friend class LabelsRepository;
};

#undef GETTER

class RpopLabel : public Label {
public:
    RpopLabel() : Label() {
        type_ = Type::rpop;
    }
};

// inlines

inline bool Label::isUser() const {
    return type_ == Type::user;
}

inline bool Label::isSystem() const {
    return type_ == Type::system;
}

using Labels = std::vector<Label>;

} // namespace macs

